#' @title Set a background map provider
#'
#' @description
#' This function sets the map provider based on the specified provider alias. It
#' retrieves the corresponding provider and attribution information from an
#' internal database of available background map providers.
#' 
#' @param alias character specifying the alias of the map provider to use.
#' Admissible values are "ign_ortho" and "ign_plan". See details
#' @param attribution logical. Should the attribution information also be
#' provided? defaults to TRUE.
#' 
#' @return A list or character vector:
#' - If `attribution` is TRUE, returns a list containing the provider and
#' attribution
#' - If `attribution` is FALSE, returns a character vector containing the
#' provider.
#' 
#' @details If no provider is specified, the function issues a warning and sets
#' provider = "ign_ortho" by default. Available aliases are:
#' - "ign_ortho": Orthophotos from IGN (Institut national de l'information
#' géographique et forestière)
#' - "ign_plan": Plan IGN v2 (Cartographie multi-échelles sur le territoire
#' national français de l'Institut national de l'information géographique et
#' forestière)
#' - "ign_parcellaire": cadastral plots from IGN (Cartographie parcellaire
#' cadastral multi-échelles sur le territoire national français de l'Institut
#' national de l'information géographique et forestière)
#' - "ign_3d": IGN's 3D representation using shading derived from MNH LiDAR HD (
#' estompage issu du MNH LiDAR HD de l'Institut national de l'information
#' géographique et forestière)
#' - "ign_building": BD TOPO® V3 - Buildings layer (BD TOPO® V3 - Bâtiments de
#' l'Institut national de l'information géographique et forestière)
#' - "ign_rpg": French land parcel identification system (RPG Dernière édition
#' de l'Institut national de l'information géographique et forestière)
#' - "ign_aoc_viti": IGN's AOC wine region boundaries (délimitations
#' parcellaires AOC viticoles (INAO) de l'Institut national de l'information
#' géographique et forestière)
#' 
#' @family interactive mapping
#' 
#' @export
#'
#' @examples
#'
#' # issues warning and returns provider "ign_ortho" and attribution information
#' set_provider() 
#' 
#' # returns provider "ign_ortho" and attribution information without warning
#' set_provider("ign_ortho")
#' 
#' # Sets the provider to "ign_plan" and returns only the provider
#' set_provider("ign_plan", attribution = FALSE)  
#' 

set_provider = function(
    alias,
    attribution = TRUE
) {
    if(missing(alias)) {
        alias = "ign_ortho"
        warning(sprintf("Provider alias missing. Please see documentation.\nChoosing '%s' by default", alias))
    }

    if(isTRUE(attribution)){
        return(
            list(
                provider = map_provider[
                    map_provider[["alias"]] == alias, 
                    "provider"
                ],
                attribution = map_provider[
                    map_provider[["alias"]] == alias, 
                    "attribution"
                ]
            )
        )
    } else {
        return(map_provider[map_provider[["alias"]] == alias, "provider"])
    }
    
}