## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk[["set"]](
  collapse = TRUE,
  comment = "#>",
  fig.align = "center"
)


## ----path---------------------------------------------------------------------
#> Loading required package: floodam.building
library(floodam.building)

# set up model to use example shipped with floodam
model_path = list(
  data = system.file("extdata", package = "floodam.building"),
  output = tempdir()
)


## ----setup, message = FALSE---------------------------------------------------
model = analyse_model(
  model = "adu_t",
  type = "adu",
  stage = c("load", "extract", "hydraulic"),
  path = model_path,
  verbose = FALSE
)


## ----plotmodel, fig.height = 5.6, fig.width = 7.1-----------------------------
plot(model)


## ----modhyd-------------------------------------------------------------------
head(model[["hydraulic"]][c("exchanges_close", "rooms")], 10) |> knitr::kable()


## ----limni--------------------------------------------------------------------
flood = generate_limnigraph(
  model = model,
  time = c(0, 5400, 10800),
  depth = cbind(facade_1 = c(0, 1.5, 0)),
  exposition = list(
    facade_1 = list(external = c("wall_A", "wall_B", "wall_C", "wall_D",
      "wall_E", "wall_F", "wall_G", "wall_H"))
  )
)


## ----plotlimni, echo = FALSE, fig.height=5.6, fig.width=7.1-------------------
plot(flood[["limnigraph"]][, "time"] / 3600, 
  flood[["limnigraph"]][, "facade_1"], xlab = "time (h)", ylab = "depth (m)",
  main = "Limnigraph", type = "l", col = rgb(1, 0, 0), lwd = 2)


## ----analyseh-----------------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic"),
  sim_id = "integrated_model"
)


## ----hydraulic_output---------------------------------------------------------
head(hydraulic[["h"]], 10) |> knitr::kable()


## ----depth, fig.height=5.6, fig.width=7.1-------------------------------------
#> visualization of the floodwater depth
plot(hydraulic)


## ----depth2, eval=FALSE-------------------------------------------------------
# #> visualization of the floodwater depth
# plot(hydraulic, view = "height")


## ----level, fig.height=5.6, fig.width=7.1-------------------------------------
#> visualization of the floodwater discharge in the openings
plot(hydraulic, view = "level")


## ----discharge, fig.height=5.6, fig.width=7.1---------------------------------
#> visualization of the floodwater discharge in the openings
plot(hydraulic, view = "discharge")


## ----openings-----------------------------------------------------------------
head(model[["data_table"]][["opening"]][c("room", "name")], 10) |> knitr::kable()


## ----depthsel, fig.height=5.6, fig.width=7.1----------------------------------
#> visualization of the floodwater depth
plot(hydraulic, selection = c("room_1", "facade_1"))


## ----pdf, fig.height=5.6, fig.width=7.1---------------------------------------
#> visualization of the floodwater discharge in the openings
plot(hydraulic, view = "height", device = "pdf", output_path = model[["path"]][["model_output_hydraulic"]])


## ----hydraulic_save-----------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic", "graph", "save"),
  what = c("h", "eQ"),
  sim_id = "integrated_model"
)


## ----analysehdam--------------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic", "damaging")
)


## ----dam----------------------------------------------------------------------
hydraulic[["damage"]] |> knitr::kable()


## ----limnimulti---------------------------------------------------------------
flood = generate_limnigraph(
  model = model,
  time = c(0, 5400, 10800),
  depth = cbind(facade_1 = c(0, 2, 0),
    facade_2 = c(0, 0, 0)),
  exposition = list(
    facade_1 = list(external = c("wall_A")),
    facade_2 = list(external = c("wall_B", "wall_C", "wall_D", "wall_E", "wall_F", "wall_G", "wall_H"))
  )
)


## ----analysehmultilim, fig.width = 7, fig.height = 7--------------------------
hydraulics = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic", "display")
)


## ----modelbase, message = FALSE, fig.width = 7, fig.height = 9----------------
model_basement = analyse_model(
  model = "adu_t_basement",
  type = "adu",
  stage = c("load", "extract", "hydraulic", "display"),
  path = model_path
)


## ----limnimultidwelling-------------------------------------------------------
flood = generate_limnigraph(
  model = model_basement,
  time = c(0, 5400, 10800),
  depth = cbind(facade_1 = c(0, 1.5, 0)),
  exposition = list(
    facade_1 = list(external_groundfloor = c("wall_A", "wall_B", "wall_C", "wall_D", "wall_E", "wall_F",
      "wall_G", "wall_H"),
      external_basement = c("wall_A", "wall_B", "wall_C", "wall_D")))
)


## ----analysehbase-------------------------------------------------------------
hydraulic_basement = analyse_hydraulic(
  model = model_basement,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic")
)


## ----plotbaseh, fig.width = 7, fig.height = 7---------------------------------
plot(hydraulic_basement, view = "level")

