## ----setup--------------------------------------------------------------------
library(floodam.building)
library(parallel)


## ----model, message = FALSE---------------------------------------------------
#> set up model path shipped with floodam.building
model_path = list(
  data = system.file("extdata", package = "floodam.building"),
  output = tempdir()
)

model = analyse_model(
  model = "adu_t",
  type = "adu",
  stage = c("load", "extract", "hydraulic", "damaging"),
  path = model_path
)

dam_less_48 = model[["damaging"]][["absolute"]][["total"]][,"12"]


## ----expdesign, message = FALSE-----------------------------------------------
hpeaks = seq(0, 2.0, by = 0.1)

floods = lapply(
  hpeaks, 
  function(h) {
    generate_limnigraph(
      model = model,
      time = c(0, 5400, 10800),
      depth = cbind(facade_1 = c(0, h, 0)),
      exposition = list(
        facade_1 = list(
          external = c(
            "wall_A", "wall_B", "wall_C", "wall_D",
            "wall_G", "wall_H", "wall_E", "wall_F"
          )
        )
      )
    )
  }
)

names(floods) = paste0("sim_", hpeaks)


## ----multisim-----------------------------------------------------------------
hydraulic = parallel::mclapply(
  names(floods), 
  function(f) {
    analyse_hydraulic(
      model = model,
      limnigraph = floods[[f]],
      opening_scenario = "close",
      stage = c("hydraulic", "damaging"),
      detail = FALSE,
      sim_id = f
    )
  }, 
 mc.cores = detectCores()/2
)


## ----multidam-----------------------------------------------------------------
damage = sapply(hydraulic, function(sim) sim[["damage"]][, "total"])


## ----plotmulti, fig.width = 7, fig.height = 7---------------------------------
plot(
  hpeaks * 100, dam_less_48[as.numeric(names(dam_less_48)) <= max(hpeaks * 100)], 
  col = "black", 
  type = "l", 
  xlab = "water height (cm)",
  ylab = "damage (euro)", 
  main = "Damage functions for short flood durations"
)
lines(hpeaks * 100, damage, type = "l", col = "red")
legend(
  "bottomright", 
  legend = c("12h ref", "simulations"), 
  col = c("black", "red"), 
  lwd = 2.0, 
  title = "flood duration (h)"
)

