#' Update elementary function damages.
#' 
#' @param path list of path with at least item damaging (where to find input data) and damaging_output (where to save output data)
#' @param hazard_range a list of value where to calculate damaging functions
#' @param d_select selected values of durations where to make some focus. At list one value.
#' @param graph logical, default to getOption("floodam_building_graph")
#' @param verbose logical, default to getOption("floodam_building_verbose")
#'
#' @return a list of array giving detailed damaging function
#' 
#' @export

update_elementary_damaging = function(
		path = getOption("floodam_building_path"),
		hazard_range = getOption("floodam_building_hazard_range"),
		d_select = getOption("floodam_building_d_select"),
		graph = getOption("floodam_building_graph"),
		verbose = getOption("floodam_building_verbose")
	)
{
	log_file = file.path(path[["damaging_output"]], "damaging.log")
	
	# Loading data
	category_vulnerability = utils::read.csv2(
		file.path(path[["damaging"]], "category_vulnerability.csv"), 
		stringsAsFactors = FALSE
	)

	category_vulnerability_complex = utils::read.csv2(
		file.path(path[["damaging"]], "category_vulnerability_complex.csv"), 
		stringsAsFactors = FALSE
	)

	component_elementary = utils::read.csv2(
		file.path(path[["damaging"]], "component_elementary.csv"), 
		stringsAsFactors = FALSE
	)
	component_elementary_action = utils::read.csv2(
		file.path(path[["damaging"]], "component_elementary_action.csv"), 
		stringsAsFactors = FALSE
	)

	# Getting elevator components
	elevator = prepare_elevator_data()

	# Checking whether merge can be done and if so merging data (rbind)
	if (isTRUE(all(names(category_vulnerability) %in% names(elevator)))) {
		category_vulnerability = rbind(
			category_vulnerability,
			elevator[names(category_vulnerability)]
		)
	} else {
		warning(
			"All column names of 'category_vulnerability' could not be found ",
			"in 'elevator'. The vulnerability categories for elementary ", 
			"components of elevator have not been added."
		)
	}
	
	if (isTRUE(all(names(component_elementary) %in% names(elevator)))) {
		component_elementary = rbind(
			component_elementary,
			elevator[names(component_elementary)]
		)
	} else {
		warning(
			"All column names of 'component_elementary' could not be found ",
			"in 'elevator'. The elementary components of elevator have not ",
			"been added."
		)
	}
	
	if (isTRUE(all(names(component_elementary_action) %in% names(elevator)))) {
		component_elementary_action = rbind(
			component_elementary_action, 
			elevator[names(component_elementary_action)]
		)
	} else {
		warning(
			"All column names of 'component_elementary_action' could not be ",
			"found in 'elevator'. The actions for elementary components of ",
			"elevator have not been added."
		)
	}

	# Processing databases
	result = lapply(
		split(
			component_elementary,
			component_elementary[["component_elementary"]]
		), 
		produce_elementary_damaging, 
		category_vulnerability = category_vulnerability,
		category_vulnerability_complex = category_vulnerability_complex, 
		component_elementary_action = component_elementary_action, 
		hazard_range = hazard_range,
		graph = graph,
		d_select = d_select,
		path = path, 
		verbose = verbose
	)
	
	names(result) = sapply(result, attr, "component_elementary")
	result = result[sort(names(result))]
	
	# Some checking
#~ 	stock = names(result)[sapply(result, function(x){attr(x,"component") == "stock"})]
#~ 	equipment = names(result)[sapply(result, function(x){attr(x,"component") == "equipment"})]
#~ 	na_value = damaging_value[is.na(damaging_value[["v_min"]]), "component_elementary"]

	write(
		sprintf(
			"Elementary damaging functions have been updated on %s.\n",
			Sys.Date()
		),
		log_file
	)

#~ 	msg = NULL
#~ 	if (length(stock[!(stock %in% damaging_value[["component_elementary"]])])) {
#~ 		warn = "Some stock are not present in damaging-value!"
#~ 		warning(sprintf("%s See %s", warn, log_file))
#~ 		msg = c(msg, warn, paste("\t-", stock[!(stock %in% damaging_value[["component_elementary"]])]), "")
#~ 	}
#~ 	if (length(equipment[!(equipment %in% damaging_value[["component_elementary"]])])) {
#~ 		warn = "Some equipment are not present in damaging-value!"
#~ 		warning(sprintf("%s See %s", warn, log_file))
#~ 		msg = c(msg, warn, paste("\t-", equipment[!(equipment %in% damaging_value[["component_elementary"]])]), "")
#~ 	}		
#~ 	if (length(stock[stock %in% na_value])) {
#~ 		warn = "Some stock in damaging-value don't have value!"
#~ 		warning(sprintf("%s See %s", warn, log_file), "")
#~ 		msg = c(msg, warn, paste("\t-", stock[stock %in% na_value]))
#~ 	}		
#~ 	if (length(equipment[equipment %in% na_value])) {
#~ 		warn = "Some equipment in damaging-value don't have value!"
#~ 		warning(sprintf("%s See %s", warn, log_file))
#~ 		msg = c(msg, warn, paste("\t-", equipment[equipment %in% na_value]), "")
#~ 	}		
#~ 	if (!is.null(msg)) write(msg, log_file, append = TRUE)
	
	msg = NULL
	for (dam in names(result)) {
		dam_ce = apply(result[[dam]], 1:2, sum, na.rm = TRUE)
		if (any(apply(dam_ce, 2, is.unsorted, na.rm = TRUE))) {
			msg = c(msg, sprintf("\t- %s", dam))
		}
	}
	if (!is.null(msg)) {
		warn = "Some elementary components do not have growing damaging functions!"
		warning(sprintf("%s See %s", warn, log_file))		
		write(c(warn, msg), log_file, append = TRUE)
	}
	
	invisible(result)
}

