#' Update damaging functions for activities at ape_05 level
#' 
#' @param type character, type of model that is loaded, default to NULL
#' @param path list of useful paths, default to getOption("floodam_building_path")
#' @param graph boolean, will graphs be plotted, default to getOption("floodam_building_graph")
#' @param verbose boolean, will floodam tells what it is doing, default to getOption("floodam_building_verbose")
#' @return NULL
#' @examples
#' \dontrun{
#' update_ape(type = "activity")
#' }
#' 
#' @export

update_ape = function(
		type, path = getOption("floodam_building_path"),
		graph = getOption("floodam_building_graph"),
		verbose = getOption("floodam_building_verbose")
	)
{
	
	path = update_path(type = type, path = path)
	activity = floodam.building::ACTIVITY
	
	# Preparing statistics
	statistic = activity[["esane"]][activity[["esane"]][["type"]] == type,]
		
	selection = dir(path[["category_output"]])
	selection = selection[file.info(file.path(path[["category_output"]], selection))[["isdir"]]]
	unlink(path[["extrapolation_output"]], recursive = TRUE, force = TRUE)
	lapply(selection, update_ape_damaging, statistic = statistic, path = path, graph = graph, verbose = verbose)
	
	selection = dir(path[["extrapolation_output"]])
	selection = selection[file.info(file.path(path[["extrapolation_output"]], selection))[["isdir"]]]
	damaging = lapply(file.path(path[["extrapolation_output"]], selection, "damaging.csv"), utils::read.csv2)
	
	marine = utils::read.csv2(file.path(path[["data"]], "marine", "marine.csv"))
	salted = 1.15											# Corresponds to min of marine	
	insee_building = 106.8 / 104.8 		# ICC 2016-T2 / 2011-T2
	insee_furniture = 100.18 / 99.46	# INSEE:001643154 2016 / 2013
	
	result = list(
		surface_building = insee_building * rbind(
			sapply(damaging, function(x) {x[["surface_building"]]}), 					# not salted
			sapply(damaging, function(x) {x[["surface_building"]]}) * salted # salted
		),
		employee_furniture = insee_furniture * rbind(
			sapply(damaging, function(x) {x[["employee_equipment"]]}) + sapply(damaging, function(x) {x[["employee_stock"]]}),	# not salted
			sapply(damaging, function(x) {x[["employee_equipment"]]}) + sapply(damaging, function(x) {x[["employee_stock"]]})		# salted (no influence)
		),											
		employee_total = rbind(
			insee_building * sapply(damaging, function(x) {x[["employee_building"]]}) + 
				insee_furniture * sapply(damaging, function(x) {x[["employee_equipment"]]}) + 
				insee_furniture * sapply(damaging, function(x) {x[["employee_stock"]]}),					# not salted
			insee_building * sapply(damaging, function(x) {x[["employee_building"]]}) * salted + 
				insee_furniture * sapply(damaging, function(x) {x[["employee_equipment"]]}) + 
				insee_furniture * sapply(damaging, function(x) {x[["employee_stock"]]})						# salted
		)
	)
	
	result = lapply(result, function(x) {
			colnames(x) = selection
			d_min = 48
			x = as.data.frame(cbind(
				h = rep(damaging[[1]][["h"]], 2),
				d = rep(damaging[[1]][["d"]], 2),
				marine = rep(c(0, 1), each = length(damaging[[1]][["h"]])),
				x
			))
			x = x[x[["d"]] > 0, ]
			x = stats::aggregate(
				x[-(1:2)],
				by = list(h = x[["h"]], d = x[["d"]] <= d_min, marine = x[["marine"]]),
				mean,
				na.rm = TRUE
			)
			x[selection] = signif (x[selection])
			
			h_step = diff(x[["h"]])[1]
			
			x[["h_min"]] = x[["h"]] - ceiling(h_step / 2)
			x[["h_max"]] = x[["h_min"]] + h_step - 1
			x[["h_min"]][x[["h"]] == 0] = -Inf
			x[["h_max"]][x[["h"]] == 0] = 0
			x[["h_max"]][which(x[["h"]] == max(x[["h"]]))] = +Inf
			x[["h_min"]][which(x[["h"]] == min(x[["h"]][x[["h"]] > 0]))] = 1
			
			x[["d_min"]] = 0
			x[["d_min"]][!x[["d"]]] = d_min + 1
			x[["d_max"]] = d_min
			x[["d_max"]][!x[["d"]]] = +Inf
			
			x[["d"]] = x[["h"]] = NULL
			return(x[order(x[["marine"]], x[["d_min"]], x[["h_min"]]), c("h_min", "h_max", "d_min", "d_max", "marine", selection)])
		})
	
	mapply(
		utils::write.csv2,
		x = result,
		file = file.path(path[["output"]], "extrapolation", paste(names(result), "csv", sep = ".")),
		row.names = FALSE
	)

	threshold = c("APE_05", "threshold_001_5", "n_total", "n_001_5", "p_001_5")	
	result[["employee_threshold"]] = statistic[statistic[["APE_05"]] %in% selection, threshold]
	result[["employee_sirene"]] = activity[["employee"]]
	
	writexl::write_xlsx(
		result,
		path = file.path(path[["output"]], "extrapolation", "activity.xlsx")
	)
	tmp = getwd()
	setwd(file.path(path[["output"]], "extrapolation"))
	command = sprintf("soffice --headless --convert-to ods %s", "activity.xlsx")
	system(command)
	setwd(tmp)
	
	invisible(NULL)
}
