#' plot a model.
#' 
#' @param x an object of class model
#' @param device, character, defines way of plotting. Three options: "display", "pdf" or "png". Default to "display".
#' @param view, vector of character, defines which view shall be plotted. Three options: "top", "front" or "damaging". Default to "top" and "front".
#' @param finition, character, defines type of finishing for plotting. Two options: "check" or "presentation". Default to "check".
#' @param option, character, defines what type of damaging will be plotted. Two options: "all" or "total". Default to "all".
#' @param selection, vector for selecting buildings to be displayed. Default to NULL: all buildings are selected.
#' @param ... some extra parameters (unused)
#' @return Nothing
#' 
#' @export

plot.model = function(
		x,
		device = "display",
		view = c("top", "front"),
		finition = "check",
		option = "all",
		selection = NULL,
		...
	)
{
	if (x[["building"]][["general"]][["version"]] == "version_03") {
		return(plot_model_03(x, device, view, finition, option, selection, ...))
	}
	if (x[["building"]][["general"]][["version"]] <= "version_02") {
		return(plot_model_02(x, device, view, finition, option, ...))
	}
	stop(sprintf("No plot.model implementation available for version %s of building", x[["building"]][["general"]][["version"]]))
}

plot_model_03 = function(
		x,
		device = "display",
		view = c("top", "front"),
		finition = "check",
		option = "all",
		selection = NULL,
		...
	)
{
	# Some checks on option
	view = unique(view)
	device = device[[1]]	
	finition = finition[[1]]
	
	if (any(is.na(match(device, c("pdf", "png", "display"))))) stop("Bad options for device")
	if (any(is.na(match(view, c("top", "front", "damaging"))))) stop("Bad options for view")
	if (any(is.na(match(finition, c("check", "presentation"))))) stop("Bad options for finition")
	if (any(is.na(match(option, c("all", "total"))))) stop("Bad options for option")
	
	if (device == "display") view = view[[1]]

	msg = c()
	if ("top" %in% view) {
		msg = c(
			msg, 
			plot_building_top_03(
				data_table = x[["data_table"]],
				model_path = x[["path"]],
				device = device,
				finition = finition,
				selection = selection,
				...
			)
		)
	}
	if ("front" %in% view) {
		msg = c(
			msg, 
			plot_building_front_03(
				data_table = x[["data_table"]],
				model_path = x[["path"]],
				device = device,
				finition = finition,
				selection = selection,
				...
			)
		)
	}
	if ("damaging" %in% view) {
		if (device != "display") {
			file_name = if (option == "total") sprintf("%s-damaging-total.pdf", x[["name"]]) else sprintf("%s-damaging.pdf", x[["name"]])
			msg = c(
				msg,
				plot.damaging(
					x[["damaging"]][["absolute"]],
					path = x[["path"]][["model_output_damaging"]],
					file_name = file_name,
					option = option
				)
			)
		} else {
			plot.damaging(x[["damaging"]][["absolute"]], option = option)
		}
	}
	
	if (device != "display") return(msg)
}	

plot_model_02 = function(
		x,
		device = "display",
		view = c("top", "front"),
		finition = "check",
		option = "all",
		...
	)
{
	# Some checks on option
	view = unique(view)
	device = device[[1]]	
	finition = finition[[1]]
	
	if (any(is.na(match(device, c("pdf", "png", "display"))))) stop("Bad options for device")
	if (any(is.na(match(view, c("top", "front", "damaging"))))) stop("Bad options for view")
	if (any(is.na(match(finition, c("check", "presentation"))))) stop("Bad options for finition")
	if (any(is.na(match(option, c("all", "total"))))) stop("Bad options for option")
	
	if (device == "display") view = view[[1]]
	
	msg = c()
	if ("top" %in% view) {
		msg = c(
			msg,
			plot_building_top_02(
				data_table = x[["data_table"]],
				model_path = x[["path"]],
				device = device,
				finition = finition,
				...
			)
		)
	}
	if ("front" %in% view) {
		msg = c(
			msg,
			plot_building_front_02(
				data_table = x[["data_table"]],
				model_path = x[["path"]],
				device = device,
				finition = finition,
				...
			)
		)
	}
	if ("damaging" %in% view) {
		if (device != "display") {
			file_name = if (option == "total") sprintf("%s-damaging-total.pdf", x[["name"]]) else sprintf("%s-damaging.pdf", x[["name"]])
			msg = c(
				msg,
				plot.damaging(
					x[["damaging"]][["absolute"]],
					path = x[["path"]][["model_output_damaging"]],
					file_name = file_name,
					option = option
				)
			)
		} else {
			plot.damaging(x[["damaging"]][["absolute"]], option = option)
		}
	}
	
	if (device != "display") return(msg)
}	
