#' Plot efficiency function
#' 
#' @param x array giving damaging function
#' @param d_select selected values of durations where to make some focus. At list one value
#' @param h_range, height range for making the graphs. Default to NULL, and so analyzing dam
#' @param d_range, duration range for making the graphs. Default to NULL, and so analyzing dam
#' @param path path for saving the result. Default to NULL, nothing is saved, graph is displayed
#' @param file_name, name of the elementary component processed. Default to NULL, no title
#' @param damaging_unit, unit of damaging
#' @param ... some extra parameters (unused)
#' @return NULL
#' 
#' @keywords internal

plot_efficiency = function(
		x,
		d_select = getOption("floodam_building_d_select"),
		h_range = NULL,
		d_range = NULL, 
		path = NULL,
		file_name = NULL,
		damaging_unit = "euro",
		...
	)
{
	
	if (!is.null(path)) 
	{
		if (is.null(file_name)) file_name = "efficiency.pdf"
		grDevices::pdf(file.path(path, file_name), width = 15 , height = 10)
	} else {
		default_par = graphics::par(no.readonly = TRUE)
	}

	# Prepare 4 different screens for general layout
	graphics::close.screen(all = TRUE)
	mat = rbind(c(0, 1, .9, 1), c(0, 1, .5, .95), c(0, 1, .07, .5), c(0, 1, 0, .1))
	graphics::split.screen(mat)
	  
	graphics::screen(1)
	if (!is.null(file_name)) {
		graphics::title(gsub("[_]", " ", strsplit(file_name, "-damaging")[[1]][1]), cex.main = 1.5)
	}

	# Deal with general graphic, prepare 3 subscreens in screen 2
	mat = rbind(c(0, .45, 0, 1), c(0.45, .5, 0, 1), c(.5, 1, 0, 1))
	graphics::split.screen(mat, screen = 2)

	# Plot the total damage as an image
	graphics::screen(5)
	xlim = if (is.null(h_range)) range(as.numeric(rownames(x[["total"]]))) else h_range
	ylim = if (is.null(d_range)) range(as.numeric(colnames(x[["total"]]))) else d_range
	suppressWarnings(
		fields::image.plot(
			x = as.integer(rownames(x[["total"]])), 
			y = as.integer(colnames(x[["total"]])), 
			z = x[["total"]], 
			col = grDevices::colorRampPalette(c("blue", "red"))(10),  #gray((10:0)/10), 
			xlab = "h (cm)", ylab = "d (hours)",
			xlim = xlim, ylim = ylim,
			zlim = c(min(0, min(x[["total"]], na.rm = TRUE)), max(1, max(x[["total"]], na.rm = TRUE))),
			main = "Full space definition"
		)
	)

	# Plot the total damage, one line by duration
	graphics::screen(7)
	plot(
		NA,
		xlim = range(as.integer(rownames(x[["total"]]))), 
		ylim = range(min(0, x[["total"]]), max(1, apply(x[["total"]], 1:2, sum, na.rm = TRUE))), 
		xlab = "h (cm)",
		ylab = sprintf("efficiency (%s)", damaging_unit),
		main = "Total"
	)
	col = grDevices::rainbow(length(d_select))
	lapply(
		as.character(d_select), 
		function(z) {
			graphics::lines(
				as.integer(rownames(x[["total"]])), x[["total"]][ , z], 
				type = "l", col = col[which(d_select == z)]
			)
		}
	)
	graphics::legend("topleft", title = "Duration", legend = paste(d_select, "hours"), lwd = 1, seg.len = .5, col = col)
	graphics::close.screen(5:7)
	
	# Plot damage by component, one plot by duration
	graphics::split.screen(c(1, length(d_select)), screen = 3)
	col = c("total" = "black", "building" = "blue", "equipment" = "red", "stock" = "green", "furniture" = "red")
  
	for (i in seq(length(d_select))) {
		graphics::screen(i + 4)
		plot(NA,
			xlim = range(as.integer(rownames(x[["total"]]))), 
			ylim = range(x[["total"]]), 
			xlab = "h (cm)",
			ylab = sprintf("efficiency (%s)", damaging_unit),
			main = sprintf("Duration: %s hours", d_select[i])
		)
		
		for (a in names(x)) {
		  graphics::lines(as.integer(rownames(x[["total"]])), x[[a]][ , as.character(d_select[i])], type = "l", col = col[a])
		}
		profit = vector("numeric",nrow(x[["total"]]))
		graphics::lines(as.integer(rownames(x[["total"]])), profit, type = "l", lty = 2, col = "gold")
	}
	
	# Plot legend of damage by actions, one legend for all
	graphics::screen(4)
	graphics::legend("center", legend = names(x), lwd = 1, col = col[names(x)], horiz = TRUE)
	
	graphics::close.screen(all = TRUE)

	if (!is.null(path)) {
		grDevices::dev.off()
		return(sprintf("Damaging function saved in %s", file.path(path, file_name)))
	} else {
		graphics::par(default_par)
		invisible(NULL)
	}	
}
