#' Plot analysis of damaging function
#' 
#' @param damaging list of list of damaging functions
#' @param what what damaging function should be plotted
#' @param group character
#' @param d_select selected values of durations where to make some focus. At list one value.
#' @param h_range, height range for making the graphs. Default to NULL, and so analyzing dam
#' @param d_range, duration range for making the graphs. Default to NULL, and so analyzing dam
#' @param output path for saving the result. Default to NULL, nothing is saved, graph is displayed.
#' @param ref character, what is the reference curve
#' @return NULL
#' 
#' @keywords internal

plot_analysis_damaging = function(
		damaging,
		what,
		group = NULL,
		d_select = getOption("floodam_building_d_select"),
		h_range = NULL,
		d_range = NULL,
		output = NULL,
		ref = NULL
	)
{	
	graphics::close.screen(all = TRUE)
  
	if (is.null(group)) group = names(damaging)
	if (length(group) == 1) group = rep(group, length(damaging))
	if (length(group) != length(damaging)) stop("length of damaging and group need to be compatible!")
	names(group) = names(damaging)
	
	if (!is.null(output)) 
	{
		grDevices::pdf(file.path(output, paste(gsub("[_]", "-", what), ".pdf", sep = "")), width = 15 , height = 10)
	}
	xlim = range(as.integer(rownames(damaging[[1]][[what]])))
	ylim = range(sapply(damaging, function(x){range(x[[what]])}))

	# Prepare 3 different screens for general layout
	mat = rbind(c(0, 1, .9, 1), c(0, 1, 0, .9))
	graphics::split.screen(mat)
	graphics::title(gsub("[_]", " ", what), cex.main = 1.5)  

	graphics::screen(2)
	mat = rbind(c(0, .6, 0, 1), c(.6, 1, 0, 1))
	graphics::split.screen(mat)

	# Plot damage by sub.curve, one plot by duration
	graphics::split.screen(c(length(d_select), 1), screen = 3)
	col = grDevices::rainbow(length(unique(group)))
	names(col) = unique(group)

	for (i in seq(length(d_select))) {
		graphics::screen(i + 4)
		plot(NA,
			xlim = xlim, 
			ylim = ylim, 
			xlab = "h (cm)", ylab = "damaging",
			main = sprintf("Duration: %s hours", d_select[i])
		)
		
		for (a in names(damaging)) {
			graphics::lines(as.integer(rownames(damaging[[1]][[what]])), damaging[[a]][[what]][ , as.character(d_select[i])], type = "l", col = col[group[a]])
		}

		if (!is.null(ref)) {
			if (length(ref) == 1) ref = rep(ref, length(rownames(damaging[[1]][[what]])))
			if (length(ref) != length(rownames(damaging[[1]][[what]]))) stop("Problem with reference curve, bad length")
			graphics::lines(as.integer(rownames(damaging[[1]][[what]])), ref, type = "l", col = "black", lwd = 2)
		}
	}
	
	# Plot legend of damage by actions, one legend for all
	graphics::screen(4)
	graphics::legend("center", legend = unique(group), lwd = 1, col = col)
	
	graphics::close.screen(all = TRUE)
	if (!is.null(output)) grDevices::dev.off()
  
	invisible(NULL)
}
