#' @title Generate a limnigraph
#'
#' @description
#' `generate_limnigraph()` builds external limnigraphs that are used as boundary
#' conditions for interior hydraulic simulation.
#'
#' @param model an object of class model
#' @param time numeric, time steps (in seconds) at which a specific floodwater
#' 	depth is reached
#' @param depth numeric, depths (in meters) reached at time steps given by
#' 	`time`. The vectors must be named after each `facade` exposed to the flood
#' @param exposition list, contains list of `facade` exposed to the flood event.
#' 	Each `facade` is a list of named vectors of the walls exposed in the same
#' 	way. Vectors are named after the `external` they belong to in the model.
#' @param verbose boolean, will floodam tells what it is doing, default to
#' 	getOption("floodam_building_verbose")
#'
#' @details
#' **Data Structure:**
#' - `time` provides the time steps at which the floodwater depth is defined. 
#' This element sets the standard length for all limnigraph data.
#' - `depth` Can be a numeric vector of the same length as `time`(for a single
#' limnigraph) or a matrix whose number of rows is equal to the length of `time`
#' (for multiple limnigraphs). Each column of the matrix represents a different
#' limnigraph, corresponding to a specific facade.
#' - `exposition`: A list defining the walls exposed to the flood. Each element
#' of the list represents a facade, and contains a named list of walls
#' exposed to that facade.
#'
#' **Naming Conventions:**
#' - When defining facades or external walls in your model, it's crucial to
#'  include the keywords "facade" and "external" in their names (e.g.,
#'  "facade_west", "external_groundfloor").
#' - Ensure that the names used in the `exposition` list match the names of
#'  the external walls in your model.
#' 
#' #' **Error Handling:**
#' - **Incompatible lengths:** If `time` and `depth` have different lengths 
#' (number of rows), the function will throw an error. Ensure that both vectors
#' have the same length.
#' - **Mismatched names:** If the column names in `depth` do not match the names
#' in `exposition`, the function will throw an error. Verify that the names are
#' consistent.
#' - **Incorrect wall names:** If the wall names in `exposition` do not exist in
#' your building model, the function may produce unexpected results. 
#' Double-check the wall IDs.
#'
#' @return a list containing the generated limnigraph and the corresponding
#'   exposition data.
#'   - `limnigraph`: A matrix with time in the first column and floodwater depths
#'     along the building walls in the subsequent columns.
#'   - `exposition`: A list of the matching wall exposures.
#'
#' @examples
#'
#' # Limnigraph
#' limnigraph = generate_limnigraph(
#'   model = adu_t,
#'   time = c(0, 300, 900),
#'   depth = cbind(facade_1 = c(0, 2, 0),
#'                 facade_2 = c(0, 0, 0)),
#'   exposition = list(
#'     facade_1 = list(external = c("wall_A", "wall_B", "wall_C", "wall_D")),
#'	   facade_2 = list(external = c("wall_E", "wall_F", "wall_G", "wall_H"))
#'   )
#' )
#'
#' # Limnigraph simplified
#' limnigraph = generate_limnigraph(
#' 	 model = adu_t,
#'   time = c(0, 300, 900),
#'   depth = cbind(facade_1 = c(0, 2, 0)),
#'   exposition = list(
#'     facade_1 = list(external =c("wall_A", "wall_B", "wall_C", "wall_D"))
#'   )
#' )
#'
#' # declaring input and output paths
#'
#'model_basement = analyse_model(
#'  model = adu_t_basement,
#'  stage = c("load", "extract", "damaging", "hydraulic")
#')
#'
#' # create limnigraph for multi level building
#'flood = generate_limnigraph(
#'  model = model_basement,
#'  time = c(0, 450, 900),
#'  depth = cbind(facade_1 = c(0, 0.2, 0),
#'    facade_2 = c(0, 0.5, 0)),
#'  exposition = list(
#'    facade_1 = list(
#'      external_groundfloor = c("wall_A"),
#'      external_basement = c("wall_A")
#'    ),
#'    facade_2 = list(
#'      external_groundfloor = c("wall_B", "wall_C"))
#'  )
#')
#'
#' @export

generate_limnigraph = function(
		model,
		time,
		depth,
		exposition,
        verbose = getOption("floodam_building_verbose") 
	) {	
	if (verbose) message(sprintf("generating limnigraph ..."))
	if (is.null(ncol(depth))) {
		dim(depth) = c(length(depth), 1)
		colnames(depth) = "exposition"
	}
	if (!is.list(exposition) && ncol(depth) == 1) {
		exposition = list(facade = exposition)
	}
	if (length(time) != nrow(depth)) {
		stop("'time' and 'depth' do not have compatible lengths.")
	}
	if (!identical(sort(colnames(depth)), sort(names(exposition)))) {
		stop("'depth' and 'exposition' names must be matching")
	}

	# check if limnigraph is correct
	mod_w = model[["data_table"]][["wall"]]
	mod_w = mod_w[is.na(mod_w[["room_name"]]), ]

	facade_not_exposed = list()
	exposed_walls = list()

	# find exposed walls
	for (f in names(exposition)) {
		for (e in names(exposition[[f]])) {
			exposition[[f]][[e]] = unlist(as.character(mod_w[["id_wall"]][mod_w[["room"]] == e & mod_w[["wall"]] %in% exposition[[f]][[e]]]))
			if (identical(exposition[[f]][[e]], character(0))) stop("Incorrect wall names")
		}
	}

	exposed_walls = unlist(exposition)

	external_not_exposed = mod_w[["id_wall"]][!(mod_w[["id_wall"]] %in% exposed_walls)]

	if (!identical(external_not_exposed, character(0))) {
		facade_not_exposed = list(external_not_exposed = external_not_exposed)
		depth = cbind(depth, facade_not_exposed = rep(0, length(time)))
		exposition[["facade_not_exposed"]] = facade_not_exposed

		if (verbose) message(sprintf(
				" walls not exposed : %s", paste(facade_not_exposed[["external_not_exposed"]],
				collapse = " ")))
	}

    result = list(limnigraph = cbind(time, depth), exposition = c(exposition))

	if (verbose) message(sprintf(" limnigraph successfully generated"))

	return(result)
}
