#' @title Extract information of a model from building and furniture components
#' 
#' @description
#' `extract_building_information()` extracts all information at buiding levels
#' to update tabular views of what is present in the model.
#' 
#' @param model an object of class model
#' @param verbose logical, default to `getOption("floodam_building_verbose")`
#' 
#' @return A list of data.frame giving a tabular view of elements present in 
#' the object `model`.
#' 
#' @details
#' 
#' Elements returned are:
#' 
#' - `parameter`: list of data frames, giving information on some elementary
#' 	components used in the model
#' - `storey`: data frame, information at storey level.
#' - `room`: data frame, information at room level.
#' - `wall`: data frame, information at wall level.
#' - `opening`: data frame, information at opening level.
#' - `coating`: data frame, information at coating level.
#' - `furniture`: data.frame, information at furniture level. This data.frame
#' 		is taken from what already desbrided in model, and adjusted  according
#' 		to information in building.
#' 
#' @examples
#' 
#' data_table = extract_building_information(adu_t)
#'
#' @export

extract_building_information = function(
		model,
		verbose = getOption("floodam_building_verbose")
	)
{
	if (model[["building"]][["general"]][["version"]] == "version_03") {
		return(
			extract_building_information_03(
				model = model, 
				verbose = verbose
			)
		)
	}
	if (model[["building"]][["general"]][["version"]] <= "version_02") {
		return(
			extract_building_information_02(
				building = model[["building"]],
				furniture = model[["data_table"]][["furniture"]],
				category = model[["category"]],
				model_name = model[["name"]],
				file_log = file.path(model[["path"]][["model_log"]], model[["file_name"]][["log"]]),
				verbose = verbose
			)
		)
	}

	stop(
		sprintf(
			"No extract_building_information implementation available for version %s of building",
			model[["building"]][["general"]][["version"]]
		)
	)
}

extract_building_information_03 = function(
		model,
		verbose = getOption("floodam_building_verbose")
	)
{
	if (verbose) message(sprintf("Extracting building information for %s...\n", model[["name"]]))
	file_log = file.path(model[["path"]][["model_log"]], model[["file_name"]][["log"]])
	# pattern = floodam.building::BUILDING[[model[["building"]][["general"]][["version"]]]]

	# First extraction
	model[["data_table"]] = list()
	model[["data_table"]][["parameter"]] = extract_parameter_03(model[["building"]])
	model[["data_table"]][["building"]] = extract_building_03(model[["building"]])
	model[["data_table"]][["storey"]] = extract_storey_03(model[["building"]])
	model[["data_table"]][["protection"]] = extract_protection_03(model[["building"]])
	model[["data_table"]][["room"]] = extract_room_03(model[["building"]])
	model[["data_table"]][["wall"]] = extract_wall_03(model[["building"]])
	model[["data_table"]][["opening"]] = extract_opening_03(model[["building"]])
	model[["data_table"]][["coating"]] = extract_coating_03(model[["building"]])
	model[["data_table"]][["furniture"]] = model[["furniture"]]
	  
	# Adjustment
	model[["data_table"]][["storey"]] = adjust_storey_03(model[["data_table"]])
	model[["data_table"]][["room"]] = adjust_room_03(model[["data_table"]])
	model[["data_table"]][["wall"]] = adjust_wall_03(model[["data_table"]])
	model[["data_table"]][["opening"]] = adjust_opening_03(model[["data_table"]], file_log)
	model[["data_table"]][["coating"]] = adjust_coating_03(model[["data_table"]], file_log)
	model[["data_table"]][["furniture"]] = adjust_furniture_03(
		model[["data_table"]],
		model[["category"]],
		model[["building"]][["general"]][["dilapidation"]],
		file_log
	)
	
	# Computing some values at model level
	selection = grep("room", model[["data_table"]][["room"]][["room"]])
	model[["value"]][["surface"]] = max(
		tapply(
			model[["data_table"]][["room"]][["surface"]][selection],
			model[["data_table"]][["room"]][["storey"]][selection],
			sum
		)
	)
	for (what in levels(model[["data_table"]][["furniture"]][[model[["category"]][1]]])) {
		selection = model[["data_table"]][["furniture"]][[model[["category"]][1]]] %in% what
		model[["value"]][[what]] = sum(model[["data_table"]][["furniture"]][["value"]][selection], na.rm = TRUE)
	}
 
	if (verbose) message(sprintf("\t... Informations successfully extracted for %s\n", model[["name"]]))

	invisible(model[["data_table"]])
}

extract_building_information_02 = function(
		building,
		furniture = NULL,
		category = NULL,
		model_name = NULL,
		file_log = NULL,
		verbose = getOption("floodam_building_verbose")
	)
{
	if (verbose) message(sprintf("Extracting building information for '%s'...\n", model_name))
	
	version_building = building[["general"]][["version"]]
	dilapidation = building[["general"]][["dilapidation"]]
	pattern = floodam.building::BUILDING[[version_building]]

	result = list()

	result[["parameter"]] = extract_parameter_02(
		building[["element"]],
		pattern[["element"]]
	)
	result[["storey"]] = extract_storey_02(
		building = building[["building"]],
		pattern = pattern[["building"]][["storey"]],
		H_abs = building[["general"]][["H"]],
		height = building[["general"]][["height"]]
	)
	result[["room"]] = extract_room_02(
		building = building[["building"]],
		pattern = pattern[["building"]][["storey"]][["generic"]]
	) 
	result[["wall"]] = extract_wall_02(
		building = building[["building"]],
		pattern = pattern[["building"]][["storey"]][["generic"]][["wall"]],
		parameter = result[["parameter"]][["wall"]]
	)
	result[["opening"]] = extract_opening_02(
		building = building[["building"]],
		pattern = pattern[["building"]][["storey"]][["generic"]][["wall"]][["opening"]],
		parameter = result[["parameter"]][["opening"]],
		file_log = file_log
	)
	result[["coating"]] = extract_coating_02(
		building = building[["building"]],
		pattern = pattern[["building"]][["storey"]][["generic"]][["wall"]][["coating"]],
		parameter = result[["parameter"]][["coating"]],
		file_log = file_log
	)
	result[["furniture"]] = furniture
		
	# Adjustment
	result[["room"]] = adjust_room_02(result, version_building)
	result[["wall"]] = adjust_wall_02(result, building[["general"]][["height"]])
	result[["opening"]] = adjust_opening_02(result, file_log)
	result[["coating"]] = adjust_coating_02(result)
	result[["furniture"]] = adjust_furniture_02(result, dilapidation, category, file_log)

	if (verbose) {
		expected = c("parameter", "storey", "room", "wall", "opening", "coating", "furniture")
		message("\t- extracted:\n", sprintf("\t\t- %s\n", names(result)))
		missing = expected[! expected %in% names(result)]
		message("\t- missing (not found):\n", sprintf("\t\t- %s\n", missing))
		message(sprintf("\t... Informations successfully extracted for '%s'\n", model_name))
	}
 
	invisible(result)
}