#' Add the detail and costs of each new material measure
#' 
#' @param measure_cost list of the different cost to be updated with the replacement measure
#' @param material_costs additional data-frame filling cost gaps in floodam.building::COMPONENT_ELEMENTARY
#' @param reference the model without measure
#' @param component the model with adaptation measures
#' @param replacement the list from l_part with the details of one material modification
#' @param what the column of the two models to compare (column changed by the measures)
#' @param elementary the column of the two models containing the elementary components
#' @param sub_type second column of the model, needed to select a certain object (ex: window, door...)
#' @param coating Null or column of coating material, if an element (floor, wall...) has a coating, add the price of the coating
#' @param insulating Null or column of insulating material, if an element (floor, wall...) has an insulating, add the price of the insulating
#' @param render Null or column of render material, if an element (floor, wall...) has a render, add the price of the render
#' @return measure_cost, a list with the cost of prevention (only material modification) and repair (repair due to flood and modification of material at the same time) and the details
#' 
#' @keywords internal
  
elementary_adaptation_cost = function(measure_cost, material_costs, reference, component, 
  replacement, what, elementary, sub_type = NULL, coating = NULL, insulating = NULL, render = NULL) {

  #Search for diffence between reference and component due to adaptation
  if (is.null(sub_type)) {
    ind = which(as.character(component[[what]]) != as.character(reference[[what]])
                & as.character(reference[[what]]) == replacement[["origin"]]
                & as.character(component[[what]]) == replacement[["final"]])
  } else {
    ind = which(as.character(component[[sub_type]]) == replacement[["sub_type"]]
          & as.character(component[[what]]) != as.character(reference[[what]])
          & as.character(reference[[what]]) == replacement[["origin"]]
          & as.character(component[[what]]) == replacement[["final"]])
  }

  if (length(ind) != 0){ 
    
    #Compute the unit support (surface, number...)
    ind_unit = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == unique(component[[elementary]][ind]))
    cost_unit = as.character(floodam.building::COMPONENT_ELEMENTARY[["v_support"]][ind_unit])
    if (cost_unit == "surface") {
      support = sum(component[[cost_unit]][ind])
      if (replacement[["sub_type"]] %in% c("window", "french_window", "door")) {
        support = support/2 
      }
    } else if (cost_unit == "unit"){
      support = length(ind)
      if (replacement[["sub_type"]] %in% c("window", "french_window", "door")) {
        support = support/2 
      } else if (cost_unit == "length"){
        # if (replacement[["type"]] == "baseboard"){ #baseboard material and support in different tables
        #   name_room = as.character(reference[["room"]][ind])
        #   ind = setdiff(which(as.character(model[["data_table"]][["wall"]][["room"]]) %in% name_room), grep("empty", model[["data_table"]][["wall"]][["type"]]))
        #   support = sum(model[["data_table"]][["wall"]][["width"]][ind])
        # }
      }
    }
    
    
    #Find the value (v_min, v_max) of the changed elementary component
    #Final material
    ind_final = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == unique(component[[elementary]][ind]))
    if (is.na(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_final]) == TRUE) { #some values are not in floodam.building::COMPONENT_ELEMENTARY
      ind_final = which(material_costs[["measure_type"]] == unique(component[[elementary]][ind]))
      value_final = mean(c(material_costs[["v_min"]][ind_final], material_costs[["v_max"]][ind_final]))
    } else {
      value_final = mean(c(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_final],floodam.building::COMPONENT_ELEMENTARY[["v_max"]][ind_final]))
    }
    #Origin material
    ind_origin = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == unique(reference[[elementary]][ind]))
    if (is.na(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_origin]) == TRUE) { #some values are not in floodam.building::COMPONENT_ELEMENTARY
      ind_origin = which(material_costs[["measure_type"]] == unique(reference[[elementary]][ind]))
      value_origin = mean(c(material_costs[["v_min"]][ind_origin], material_costs[["v_max"]][ind_origin]))
    } else {
      value_origin = mean(c(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_origin],floodam.building::COMPONENT_ELEMENTARY[["v_max"]][ind_origin]))
    }
    
    #Presence of a coating, identical to the original coating
    value_coating = 0
    if (!is.null(coating)){
      ind_coating = intersect(ind, which(as.character(component[[coating]]) == as.character(reference[[coating]]))) #need to add coating only for element with different material and same coating
      if (length(ind_coating != 0)){
        df_coating = component[ind_coating, c(cost_unit, paste(coating, "elementary_component", sep = "_"))]
        for (i in 1:length(df_coating[,1])){
          if (!is.na(df_coating[i,2])){
            ind_c = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == as.character(df_coating[i,2]))
            value_coating = value_coating + df_coating[[cost_unit]][i] * mean(c(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_c],floodam.building::COMPONENT_ELEMENTARY[["v_max"]][ind_c]))
          }
        }
      } 
    }
    
    #Presence of an insulating, identical to the original insulating
    value_insulating = 0
    if (!is.null(insulating)){
      ind_insulating = intersect(ind, which(as.character(component[[insulating]]) == as.character(reference[[insulating]]))) #need to add insulating only for element with different material and same insulating
      if (length(ind_insulating != 0)){
        df_insulating = component[ind_insulating, c(cost_unit, paste(insulating, "elementary_component", sep = "_"))]
        for (i in 1:length(df_insulating[,1])){
          if (!is.na(df_insulating[i,2])){
            ind_i = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == as.character(df_insulating[i,2]))
            value_insulating = value_insulating + df_insulating[[cost_unit]][i] * mean(c(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_i],floodam.building::COMPONENT_ELEMENTARY[["v_max"]][ind_i]))
          }
        }
      }
    }
    # #Presence of a render, identical to the original render
    # value_render = 0
    # if (!is.null(render)){
    #   ind_render = intersect(ind, which(as.character(component[[render]]) == as.character(reference[[render]]))) #need to add render only for element with different material and same render
    #   if (length(ind_render != 0)){
    #     df_render = component[ind_render, c(cost_unit, paste(render, "elementary_component", sep = "_"))]
    #     for (i in 1:length(df_render[,1])){
    #       if (!is.na(df_render[i,2])){
    #         ind_r = which(row.names(floodam.building::COMPONENT_ELEMENTARY) == as.character(df_render[i,2]))
    #         value_render = value_render + df_render[[cost_unit]][i] * mean(c(floodam.building::COMPONENT_ELEMENTARY[["v_min"]][ind_r],floodam.building::COMPONENT_ELEMENTARY[["v_max"]][ind_r]))
    #       }
    #     }
    #   }
    # }
    
    #Compute the elementary cost
    cost_prevention = value_final*support + value_coating + value_insulating #+ value_render
    cost_repair = (value_final - value_origin)*support
    
    measure_cost[["prevention"]] = measure_cost[["prevention"]] + cost_prevention
    measure_cost[["repair"]] = measure_cost[["repair"]] + cost_repair
    if (cost_prevention != 0){ #Avoid too much details about the measures not realised
      if (replacement[["type"]] == "opening"){
        measure_cost[["detail"]] = rbind(measure_cost[["detail"]], data.frame(
          measure_name = paste(replacement[c("sub_type", "part")], collapse = "_"), 
          origin = replacement[["origin"]], final = replacement[["final"]], v_support = cost_unit, 
          support = support, prevention_cost = cost_prevention, repair_cost = cost_repair, crawlspace_cost = 0, slab_cost = 0))
      } else {
      measure_cost[["detail"]] = rbind(measure_cost[["detail"]], data.frame(
        measure_name = paste(replacement[c("type", "part")], collapse = "_"), 
        origin = replacement[["origin"]], final = replacement[["final"]], v_support = cost_unit, 
        support = support, prevention_cost = cost_prevention, repair_cost = cost_repair, crawlspace_cost = 0, slab_cost = 0))
      }
    }
  }
    
  
  
	return(measure_cost)
}
  
  

