#' creates a new model template
#' 
#' @param input_file character,either "building", "furniture" or "metadata", 
#'  depending on the input file you wish to create 
#' @param version_building character, version of building model to be used, 
#'  default to "02". Not necessary when input_file = "furniture"
#' @param type_building character, type of building file to create ("xml" or 
#'  "yaml"), default to xml. Not necessary when input_file = "furniture"
#' @param file_path character, path to folder where to store the input file, 
#'  default to tempdir()
#' @param editor character, the text editor to be used
#' @return new editor window with building.xml template open
#' @examples
#' \dontrun{
#' 
#' create_new_model(input_file = "building")
#' 
#' create_new_model(input_file = "building", editor = "code")
#' 
#' create_new_model(input_file = "furniture", editor = "rstudio")
#' 
#' create_new_model(
#'    input_file = "building", 
#'    type_building = "yaml", 
#'    editor = "code"
#' )
#' 
#' create_new_model(input_file = "metadata", editor = "code")
#' }
#' 
#' @export 
  
create_new_model = function(
  input_file,
  version_building = "02", 
  type_building = "xml",
  file_path = tempdir(), 
  editor = getOption("editor")
) {

  # initial parameter checkup
  warn = "Please provide a valid input file"
  if(missing(input_file)) stop(warn)
  if(!(input_file %in% c("building", "furniture", "metadata"))) stop(warn)

  # generation of the building input file
  if(input_file == "building"){
    template = switch(
      version_building, 
      "02" = building_file_template_02, 
      "03" = building_file_template_03
    )
    
    if(type_building == "xml"){
      utils::file.edit(
        XML::saveXML(
            list2xml(template), file.path(file_path, "building.xml"), 
            encoding = "UTF-8", 
            prefix = '<?xml version="1.0" encoding="UTF-8"?>\n'
          ),
        editor = editor
      )
    }
    if(type_building == "yaml"){
      yaml::write_yaml(template, file.path(file_path, "building.yaml"))
      utils::file.edit(
        file.path(file_path, "building.yaml"),
        editor = editor
      )  
    }
  }
  
  # generation of the furniture input file
  if(input_file == "furniture"){
    utils::write.csv(
      furniture_file_template,
      file = file.path(file_path, "furniture.csv"),
      row.names = FALSE, 
      quote = FALSE
    )
    utils::file.edit(
      file.path(file_path, "furniture.csv"),
      editor = editor
    )
  }

  # generation of the metadata file
  if(input_file == "metadata"){
    yaml::write_yaml(
      metadata_file_template, 
      file.path(file_path, "metadata.yaml")
    )
    utils::file.edit(
      file.path(file_path, "metadata.yaml"),
      editor = editor
    )  
  }
}
