#' Analyse the hydraulics of the model
#'
#' @param opening data.frame, containing informations on the openings, coming
#' 		from the hydraulic part of the model
#' @param room data.frame, containing informations on the rooms, coming
#' 		from the hydraulic part of the model
#' @param bound numeric, matrix of floodwater depth heights by instant t
#' 		for each exterior opening exposed to flooding
#' @param open data.frame, containing informations on the openings in an `open`
#' 		scenario, coming from the hydraulic part of the model
#' @param dt_max numeric, optional; maximum timestep for simulation
#' @param t_start numeric, start time of the simulation
#' @param t_max numeric, duration of the flood event in seconds,
#' 		optional; if not provided, calculated using limnigraph
#' @param height_break numeric, the differential height on each side of an
#' 		opening needed to break it
#' @param r_names_to_merge list of room vectors to be merged
#' @param detail boolean, if FALSE only return hmax
#'
#' @return lists of hydraulic values: water exchange through openings and
#' floodwater depth in rooms
#'
#' @export

compute_hydraulic = function(opening, room, bound, open, dt_max, t_start = 0, 
	t_max, height_break = 1.0, r_names_to_merge = NULL, detail = TRUE) {

	base_room = room
	base_room_name = base_room[["room"]]

	# merging rooms if needed
	if (!is.null(r_names_to_merge)) {
		merged_data = merge_room(opening, room, r_names_to_merge)
		opening = merged_data[["opening"]]
		room = merged_data[["room"]]
		summed_room = merged_data[["summed_room"]]
		sum_names = names(summed_room)
		room_name = room[["room"]]
		n_s_r = room_name[! room_name %in% sum_names]
	}

	# setting up names
	exterior_name = colnames(bound)[-1]
	room_name = room[["room"]]
	opening_name = opening[["opening"]]
	building_name = c(room_name, exterior_name)

	# init building and boundaries
	building = rbind(
		surf = c(room[["surface"]],	rep(NA_real_, ncol(bound) - 1)),
		vmax = c(room[["v"]], rep(NA_real_, ncol(bound) - 1)),
		z0 = c(room[["H_abs"]], numeric(ncol(bound) - 1)),
		z = c(room[["initial_depth"]], bound[1, -1]),
		v = c(room[["initial_depth"]] * room[["surface"]], rep(NA_real_, ncol(bound) - 1))
	)
	colnames(building) = building_name
	building["z", ] = building["z", ] + building["z0", ]

	if (any(room[["initial_depth"]] < 0)) stop("h cannot be negative")

	# init exchange
	exchange = rbind(
		width = opening[["width"]],
		z_bottom = opening[["H_abs"]],
		z_top = opening[["H_top"]], 
		Cd = opening[["discharge_coeff"]],
		z_up = 0,
		z_down = 0,
		flow = 0,
		s = 0
	)
	colnames(exchange) = opening_name

	# setting up the relations between rooms and exchange
	match_opening = t(as.matrix(opening[c("up", "down")]))
	colnames(match_opening) = opening_name
	matching = matrix(
		0, 
		nrow = nrow(opening),
		ncol = ncol(building),
		dimnames = list(opening_name, building_name)
	)
	for (o in opening_name) matching[o, match_opening[, o]] = c(-1, 1)

	# preallocating the output matrices
	nrows = floor(t_max / dt_max)
	result_z = matrix(
		NA_real_,
		nrow = nrows,
		ncol = ncol(building) + 1,
  		dimnames = list(c(), c("time", building_name)))
	result_q = matrix(
		NA_real_,
		nrow = nrows,
  		ncol = nrow(opening) + 1,
  		dimnames = list(c(), c("time", opening_name)))
	result_ve = result_s = result_q
	result_vo = result_z
	result_z[1, 1] = result_vo[1, 1] = result_q[1, 1] = result_s[1, 1] =
		result_ve[1, 1] = t_start
	
	pos = 1

	# declaring the interpolation functions to compute water level on the outside
	approx_fun = sapply(
		exterior_name,
		function(x) {stats::approxfun(bound[, "time"], bound[, x], yright = 0)}
	)

	# time loop
	t_calc = t_start
	for (pos in seq(nrows)) {
		# get level for boundaries
		for (e in exterior_name) {
			z = approx_fun[[e]](t_calc)
			building["z", e] = z
		}
		# get level at exchange up and down
		exchange[c("z_up", "z_down"), ] = building["z", match_opening]

		# checking opening collapse
		sel_col = pmax(exchange["z_up", ], exchange["z_down", ]) >
			exchange["z_bottom", ] + height_break &
			abs(exchange["z_up", ] - exchange["z_down", ]) > height_break
		exchange["z_top", sel_col] = open[sel_col, "H_abs"] + open[sel_col, "height"]

		# compute flow and section for all exchange
		exchange[c("s", "flow"), ] = rcpp_compute_exchange(exchange)
		# exchange[c("s", "flow"), ] = compute_exchange(exchange, opening_name)

		# check if room full
		check = room_name[building["v", room_name] > building["vmax", room_name]]
		if (length(check) > 0) {
			sel = exchange["flow", ] * matching[, check, drop = FALSE]
			op = unlist(apply(sel, 2, function(x) rownames(sel)[x > 0]))
			exchange["flow", op] = 0
		}

		# compute dt_adj if needed (case room empty)
		dt_adj = dt_max
		Qtot = colSums(exchange["flow", ] * matching[, room_name, drop = FALSE])
		check = room_name[Qtot < 0]
		if (length(check) > 0) {
			dt_adj = min(dt_adj, building["v", check] / -Qtot[check])
		}

		Qtot_adj = Qtot * dt_adj

		# get volume for room
		building["v", room_name] = building["v", room_name] + Qtot_adj

		# check room surfaces when there are merged rooms
		if (!is.null(r_names_to_merge)) {
			for (r in names(summed_room)) {
				# find room data based on water volume inside for merged rooms
				sel = building["v", r] >= summed_room[[r]][
					, "volume", drop = FALSE]
				vol = max(summed_room[[r]][, "volume"][sel])

				# get room surface based on volume for merged rooms
				building["surf", r] = summed_room[[r]][, "surface"][
					summed_room[[r]][, "volume"] == vol]

				# get water level based on volume for merged rooms
				z = summed_room[[r]][ ,"z"][summed_room[[r]][ ,"volume"] == vol]
				building["z", r] = z + (building["v", r] - vol) /
					building["surf", r]
			}

			# not merged rooms
			building["z", n_s_r] = building["z0", n_s_r] +
				building["v", n_s_r] / building["surf", n_s_r]

		} else {
			# computing level based on volume
			building["z", room_name] = building["z0", room_name] +
				building["v", room_name] / building["surf", room_name]
		}

		# updating the output matrices
		result_z[pos, ] = c(t_calc, building["z", ])
		result_q[pos, ] = c(t_calc, exchange["flow", ])
		result_s[pos, ] = c(t_calc, exchange["s", ])
		result_vo[pos, ] = c(t_calc, building["v", ])
		result_ve[pos, ] = c(t_calc, exchange["flow", ] / exchange["s", ])

		# time incrementation
		t_calc = t_calc + dt_adj
	}

	h_base = base_room[["H_abs"]]
	names(h_base) = base_room[["room"]]

	# split output when invisible walls
	if (!is.null(r_names_to_merge)) {
		result_z = split_output(result_z, r_names_to_merge, base_room)
		# result_vo = split_output(result_vo, r_names_to_merge, base_room)
	}

	# cut when h > ceiling_H
	ceiling = base_room[["H_abs"]] + base_room[["ceiling_H"]] / 100
	names(ceiling) = base_room[["room"]]
	result_z[, names(ceiling)] = pmin(result_z[, names(ceiling)],
        rep(ceiling, each = nrow(result_z)))

	result_h = result_z
	# convert water level to water height
	result_h[, names(h_base)] = result_z[, names(h_base)] -
		rep(h_base, each = nrow(result_z))
	result_h = pmax(result_h, 0)

	# find max water level by room
	building_name = c(base_room_name, colnames(bound)[-1])
	hmax = rbind(
		z = numeric(length(building_name)),
		t = numeric(length(building_name))
	)
	colnames(hmax) = building_name
	hmax["z", ] = sapply(building_name, function(r) max(result_h[, r]))
	hmax["t", ] = sapply(building_name, function(r) ifelse(hmax["z", r] != 0.0,
		result_h[result_h[, r] == max(result_h[, r]), "time"], 0.0))
	hmax = cbind(hmax, mean = c(mean(hmax["z", ][base_room_name]),
		mean(hmax["t", ][base_room_name])))

	# returning output
	if (isFALSE(detail)) {
		return(
			list(
				"hmax" = hmax
			)
		)
	} else {
		return(
			list(
				"h" = result_h,
				"z" = result_z,
				"eQ" = result_q,
				"eS" = result_s,
				"ve" = result_ve,
				"vo" = result_vo,
				"hmax" = hmax
			)
		)
	}
}
