#' @title Compute surface and flow for a matrix of openings
#' 
#' @details
#' `compute_exchange()` computes wet surfaces and flows for a matrix of openings.
#' 
#' The matrix must be organized as follows:
#' - columns give informations for openings (pssibly named, butnot mandatory)
#' - rows give informations for each of those entities (names are mandatory):
#'   - `width`: width of the opening in meter
#'   - `z_bottom`: absolute altitude of the bottom of the opening in meter
#'   - `z_top`: absolute altitude of the top of the opening in meter
#'   - `Cd`: coefficient giving the loss of charge (no dimension)
#'   - `z_up`: absolute level of water in the 'upward' side of the opening in meter
#'   - `z_down`: absolute level of water in the 'downward' side of the opening in meter
#' 
#' `compute_exchange()` is the R counterpart of `rcpp_compute_exchange()`. Both functions should give
#' exactly the same numeric results.  `compute_exchange()` is used mainly for testing purposes.
#' 
#' @param exchange matrix, containing informations for openings. See details.
#'
#' @return matrix of surface and flow for each openings.
#'
#' @export

compute_exchange = function(exchange) {
	opening_name = colnames(exchange)
	if (is.null(opening_name)) opening_name = seq(ncol(exchange))

	result = exchange[c("s", "flow"), ]

	for (o in opening_name) {
		x = exchange[, o]
		if (x["z_up"] >= x["z_down"]) {
			z_up = x["z_up"]		# upstream
			z_down = x["z_down"]	# downstream
			direction = 1			# direction
		} else {
			z_up = x["z_down"]		# upstream
			z_down = x["z_up"]		# downstream
			direction = -1			# direction
		}

		if (z_up <= x["z_bottom"]) {
			# no flow: water level lower to crest elevation
			# case = 0
			s = 0.0
			flow = 0.0
		} else {
			# flow
			# case += 1
			if (z_up <= x["z_top"]) {
				# cas += 10
				if (z_down <= x["z_bottom"]) {
					# unsubmerged weir law
					# case += 100
					s = x["width"] * (z_up - x["z_bottom"])
					flow = x["Cd"] * sqrt(2 * 9.81) * s * sqrt(z_up - x["z_bottom"])
				} else {
					# submerged weir law
					# case += 200
					s = x["width"] * ((z_up + z_down) / 2 - x["z_bottom"])
					flow = x["Cd"] * sqrt(2 * 9.81) * s * sqrt(z_up - z_down)
				}
			} else {
				# case += 20
				if (z_down <= x["z_bottom"]) {
					# unsubmerged gate law
					# case += 100
					s = x["width"] * (x["z_top"] - x["z_bottom"])
					flow = x["Cd"] * sqrt(2 * 9.81) * s * sqrt(z_up - (x["z_bottom"] + x["z_top"]) / 2)
				} else {
					# submerged gate law
					# case += 200
					s = x["width"] * (x["z_top"] - x["z_bottom"])
					flow = x["Cd"] * sqrt(2 * 9.81) * s * sqrt(z_up - z_down)
				}
			}
		}

		# directing flow
		result[, o] = c(s, direction * flow)
	}
	result
}
