#' @keywords internal 

compute_damage_detail = function(
    data_table,
    dam,
    hazard_range,
    h_abs = c("model", "room", "component"),
    file_log,
    dilapidation,
    version_building,
    talkative = FALSE
) {
    h_abs = match.arg(h_abs)
    if (h_abs %in% c("room", "component")) data_table = adjust_h_abs(data_table, h_abs)

	result = list()
	
	# Room
	stage("\t1. Compute damage for room")

	stage("\t\tPreparing ceiling")
	ceiling = data_table[["room"]][
        c(
            "room",
            "storey",
            "ceiling_elementary_component", 
            "ceiling_insulating_elementary_component",
            "ceiling_coating_elementary_component",
            "H_abs",
            "surface",
            "ceiling_H"
        )
    ]
	ceiling[["H_abs_room"]] = ceiling[["H_abs"]]
	ceiling[["H_abs"]] = ceiling[["H_abs"]] + ceiling[["ceiling_H"]]

	stage("\t\tStart adjusting elementary functions for room")
	result[["room"]] = c(
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["room"]],
            what = "floor_elementary_component",
            surface_height = NULL,
            dilapidation = 0,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["room"]],
            what = "floor_coating_elementary_component",
            surface_height = NULL,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["room"]],
            what = "floor_insulating_elementary_component",
            surface_height = NULL,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = ceiling,
            what = "ceiling_elementary_component",
            surface_height = NULL,
            dilapidation = 0,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = ceiling,
            what = "ceiling_insulating_elementary_component",
            surface_height = NULL,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = ceiling,
            what = "ceiling_coating_elementary_component",
            surface_height = NULL,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        )
    )
	
    stage("\t\tPost-treatment for room")
	result[["room"]] = result[["room"]][!sapply(result[["room"]], is.null)]

	# Walls
	stage("\t2. Compute damage for wall")
	
    stage("\t\tCompute surface_height")
	surface_height = mapply(
        calculate_surface_height,
		wall = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
		MoreArgs = list(
            opening = data_table[["opening"]],
            version_building = version_building
        )
    )

	stage("\t\tStart adjusting elementary functions for wall")
	result[["wall"]] = c(
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["wall"]],
            what = "wall_elementary_component",
            surface_height = surface_height,
            dilapidation = 0,  # No distinction between partition and load_bearing_wall (as done by CEPRI 2013)
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["wall"]],
            what = "insulating_elementary_component",
            surface_height = surface_height,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        ),
        adjust_dam_by_component(
            dam = dam,
            component = data_table[["wall"]],
            what = "render_elementary_component",
            surface_height = surface_height,
            dilapidation = dilapidation,
            hazard_range = hazard_range,
            file_log = file_log
        )
	)

	stage("\t\tPost-treatment for wall")
	result[["wall"]] = result[["wall"]][!sapply(result[["wall"]], is.null)]
	selection = grep("partition", names(result[["wall"]]))
	selection_names = grep("partition", names(result[["wall"]]), value = TRUE)
	result[["wall"]][selection] = lapply(result[["wall"]][selection], function(x){x / 2}) 
	names(result[["wall"]][selection]) = selection_names

	# Openings
    stage("\t3. Compute damage for opening")
    if (!is.null(data_table[["opening"]])) {
        stage("\t\tStart adjusting elementary functions for opening")
        result[["opening"]] = c(
            adjust_dam_by_component(
                dam = dam,
                component = data_table[["opening"]],
                what = "opening_elementary_component",
                surface_height = NULL,
                dilapidation = dilapidation,
                hazard_range = hazard_range,
                file_log = file_log
            ),
            adjust_dam_by_component(
                dam = dam,
                component = data_table[["opening"]],
                what = "shutter_elementary_component",
                surface_height = NULL,
                dilapidation = dilapidation,
                hazard_range = hazard_range,
                file_log = file_log
            )
        )
     
        stage("\t\tPost-treatment for opening")
        result[["opening"]] = result[["opening"]][!sapply(result[["opening"]], is.null)]
        result[["opening"]] = lapply(result[["opening"]], function(x){x/2}) 
    }

	# Coating
    stage("\t4. Compute damage for coating")
    if (!is.null(data_table[["coating"]])) {
        stage("\t\tCompute surface_height")
        surface_height = mapply(
            calculate_surface_height,
            wall = split(data_table[["coating"]], seq(nrow(data_table[["coating"]]))),
            MoreArgs = list(
                opening = data_table[["opening"]],
                version_building = version_building
            )
        )

        stage("\t\tStart adjusting elementary functions for coating")
        result[["coating"]] = c(
            adjust_dam_by_component(
                dam = dam,
                component = data_table[["coating"]],
                what = "coating_elementary_component",
                surface_height = surface_height,
                dilapidation = dilapidation,
                hazard_range = hazard_range,
                file_log = file_log
            )
        )

        stage("\t\tPost-treatment for coating")
	    result[["coating"]] = result[["coating"]][!sapply(result[["coating"]], is.null)]
    }

	# Furniture
    stage("\t5. Compute damage for furniture")
    if (!is.null(data_table[["furniture"]])) {
        stage("\t\tStart adjusting elementary functions for furniture")
        result[["furniture"]] = c(
            adjust_dam_by_component(
                dam = dam,
                component = data_table[["furniture"]],
                what = "component_elementary",
                surface_height = NULL,
                dilapidation = 0, # Already included in furniture.csv !
                hazard_range = hazard_range,
                file_log = file_log
            )
        )

        stage("\t\tPost-treatment for furniture")
        result[["furniture"]] = result[["furniture"]][!sapply(result[["furniture"]], is.null)]
    }

	return(result)
}

#' @keywords internal
#' 
#' 

adjust_dam_by_component = function(
        dam,
        component,
        what,
        surface_height,
        dilapidation,
        hazard_range,
        file_log
    ) 
{
    if (is.null(component)) {
        return(NULL)
    }
    if (is.null(component[[what]])) {
        return(NULL)
    }
    if (is.null(surface_height)) {
        mapply(
            adjust_elementary_damaging, 
            damaging = dam[as.character(component[[what]])],
            component = split(component, seq(nrow(component))),
            MoreArgs = list(
                what = what,
                dilapidation = dilapidation,
                hazard_range = hazard_range,
                file_log = file_log),
            SIMPLIFY = FALSE
        )
    } else {
        mapply(
            adjust_elementary_damaging, 
            damaging = dam[as.character(component[[what]])],
            component = split(component, seq(nrow(component))),
            surface_height = surface_height,
            MoreArgs = list(
                what = what,
                dilapidation = dilapidation,
                hazard_range = hazard_range,
                file_log = file_log),
            SIMPLIFY = FALSE
        )
    }
}
