#' Calculate the total costs of the selected adaptation measures
#' 
#' @param model the model without adaptation measures
#' @param model_adapted the model to which measures were applied
#' @param measure the list of measures applied to the model
#' @return a vector with the total cost of the combination of measures
#' 
#' @keywords internal

compute_adaptation_costs = function(model, model_adapted, measure) {
###Create empty data frame to save the costs of each measure
  cost_detail = data.frame(measure_name = "total", origin = "/", final = "/", v_support = "/", 
       support = NA, prevention_cost = numeric(1), repair_cost = numeric(1), crawlspace_cost = numeric(1), slab_cost = numeric(1))
  
#Initialization
  measure_cost = list(prevention = 0, repair = 0, crawlspace = 0, slab = 0, detail = cost_detail)
  
###Cost of resist
  if ("resist" %in% names(measure)) {
	if ("resist_frpp" %in% measure[["resist"]][["measure_name"]]){
	resist_costs = utils::read.csv2(file.path(model_adapted[["path"]][["adaptation"]], "resist-cost-frpp.csv"), stringsAsFactors = FALSE)
	} else {
	resist_costs = utils::read.csv2(file.path(model_adapted[["path"]][["adaptation"]], "resist-cost.csv"), stringsAsFactors = FALSE)
    	}
    #Number of opening needing cofferdam
    ind_ext = grep("external", model[["data_table"]][["opening"]][["room"]])
    ind_rdc = which(-100 < model[["data_table"]][["opening"]][["H_abs"]] - model[["data_table"]][["opening"]][["H"]] & model[["data_table"]][["opening"]][["H_abs"]] - model[["data_table"]][["opening"]][["H"]] < 100)
    nb_op = length(which(model[["data_table"]][["opening"]][["opening_type"]][intersect(ind_ext, ind_rdc)] %in% c("door"))) #nb of exterior door
    nb_op = nb_op + length(which(model[["data_table"]][["opening"]][["opening_type"]] == "french_window"))/2 # add nb of french window (case of appartement_01)
    
    #Building perimeter
    ind_ext = grep("external", model[["data_table"]][["wall"]][["room"]])
    ind_rdc = which(-100 < model[["data_table"]][["wall"]][["H_abs"]] & model[["data_table"]][["wall"]][["H_abs"]] < 100)
    building_perimeter = sum(model[["data_table"]][["wall"]][["width"]][intersect(ind_ext,ind_rdc)]) # perimeter is the sum of the width of the external walls of the groundfloor
    
    #Dimensionning height
    h_dim = as.numeric(measure[["resist"]][["h_dim"]])
    
    #Add surface/unit/perimeter values
    ind = grep("unit", resist_costs[["v_support"]])
    resist_costs[["v_min"]][ind] = resist_costs[["v_min"]][ind]*nb_op
    resist_costs[["v_max"]][ind] = resist_costs[["v_max"]][ind]*nb_op
    ind = grep("surface", resist_costs[["v_support"]])
    resist_costs[["v_min"]][ind] = resist_costs[["v_min"]][ind]*building_perimeter*h_dim/100 #h_dim is in cm
    resist_costs[["v_max"]][ind] = resist_costs[["v_max"]][ind]*building_perimeter*h_dim/100 #h_dim is in cm
    ind = grep("perimeter", resist_costs[["v_support"]])
    resist_costs[["v_min"]][ind] = resist_costs[["v_min"]][ind]*building_perimeter
    resist_costs[["v_max"]][ind] = resist_costs[["v_max"]][ind]*building_perimeter
    
    measure_cost[["prevention"]] = measure_cost[["prevention"]] + sum(resist_costs[["v_min"]],resist_costs[["v_max"]])/2
    measure_cost[["repair"]] = measure_cost[["repair"]] + sum(resist_costs[["v_min"]],resist_costs[["v_max"]])/2
    
    #save values
    measure_cost[["detail"]] = rbind(measure_cost[["detail"]], data.frame(
        measure_name = "resist", origin ="/", final ="/",
        v_support = "building_perimeter", support = building_perimeter, 
        prevention_cost = measure_cost[["prevention"]], repair_cost = measure_cost[["repair"]], crawlspace_cost = measure_cost[["crawlspace"]], slab_cost = measure_cost[["slab"]]))
  }

###Cost of avoid

  if ("avoid" %in% names(measure)){
    cost = utils::read.csv2("data-floodam/adaptation/avoid-cost.csv")

    # Ratio of construction prices between the US and France
    cr = 0.97

    # Exchange rate in 2009
    er = 0.72

    # Construction price development in France between 2009 and 2017 (Decembre)
    cp = 1670 / mean(c(1507, 1502, 1498, 1503))
    
    # Conversion centimeter/feet
    cfc = 30.48
    
    # Conversion squared meter/ squared feet
    sqrt_cfm = (cfc / 100)^2

    # Convert dollars in euros 2018/m^2
    cost[["eurofoot2"]] = cost[["dollarfoot2"]] * cr * er * cp
    cost[["eurom2"]] = cost[["eurofoot2"]] / sqrt_cfm

    # Cost by centimeter of elevation
    cost_euro_m2_cm_crawlspace = (cost[["eurom2"]][cost[["feet"]] == 4 & cost[["type"]] == "crawlspace"] - cost[["eurom2"]][cost[["feet"]] == 2 & cost[["type"]] == "crawlspace"]) / (2 * cfc)
    cost_euro_m2_cm_slab = (cost[["eurom2"]][cost[["feet"]] == 4 & cost[["type"]] == "slab_on_grade"] - cost[["eurom2"]][cost[["feet"]] == 2 & cost[["type"]] == "slab_on_grade"]) / (2 * cfc)
 
    # Compute cost

    ## Dimensionning height
    h_dim = as.numeric(measure[["avoid"]][["h_dim"]])
    measure_cost[["crawlspace"]] = measure_cost[["crawlspace"]] + sum(model[["data_table"]][["room"]][["surface"]][!is.na(model[["data_table"]][["room"]][["name"]]) & model[["data_table"]][["room"]][["storey"]] == unique(model[["data_table"]][["room"]][["storey"]])[1]]) * (cost[["eurom2"]][cost[["type"]] == "crawlspace" & cost[["feet"]] == 2] + (h_dim - 2 * cfc) * cost_euro_m2_cm_crawlspace)
    measure_cost[["slab"]] = measure_cost[["slab"]] + sum(model[["data_table"]][["room"]][["surface"]][!is.na(model[["data_table"]][["room"]][["name"]]) & model[["data_table"]][["room"]][["storey"]] == unique(model[["data_table"]][["room"]][["storey"]])[1]]) * (cost[["eurom2"]][cost[["type"]] == "slab_on_grade" & cost[["feet"]] == 2] + (h_dim - 2 * cfc) * cost_euro_m2_cm_slab)

    #save values
    measure_cost[["detail"]] = rbind(measure_cost[["detail"]], data.frame(
    measure_name = "avoid", origin ="/", final ="/",
    v_support = "surface", support = sum(model[["data_table"]][["room"]][["surface"]][!is.na(model[["data_table"]][["room"]][["name"]]) & model[["data_table"]][["room"]][["storey"]] == unique(model[["data_table"]][["room"]][["storey"]])[1]]), 
    prevention_cost = measure_cost[["prevention"]], repair_cost = measure_cost[["repair"]], crawlspace_cost = measure_cost[["crawlspace"]], slab_cost = measure_cost[["slab"]]))

  }

###Cost of material
  if ("material" %in% names(measure)) {
    
  #Cost of lacking data
    material_costs = utils::read.csv2(file.path(model_adapted[["path"]][["adaptation"]], "material-cost.csv"), 
                             stringsAsFactors = FALSE)
    material_costs[["v_min"]] = as.numeric(material_costs[["v_min"]])
    material_costs[["v_max"]] = as.numeric(material_costs[["v_max"]])
    
  #Split the list of material measure by element
    l_type = split(measure[["material"]], measure[["material"]][["type"]], drop = TRUE)
    
  # Baseboard
  # Uncomment when baseboard taken into account (material et component_elementary in model[["data_table"]][["room"]])
  #   if ("baseboard" %in% names(l_type)){
  #     l_part = split(l_type[["baseboard"]], l_type[["baseboard"]][["part"]], drop = TRUE)
  #     
  #     #Material
  #     if ("material" %in% names(l_part)) {
  #       for (i in 1:nrow(l_part[["material"]])){
  #         replacement = l_part[["material"]][i,]
  #         measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["room"]], 
  #           model_adapted[["data_table"]][["room"]], replacement, "baseboard_material", "baseboard_elementary_component")
  #       }
  #     }
  #   }
    
  #Ceiling
    if ("ceiling" %in% names(l_type)){
      l_part = split(l_type[["ceiling"]], l_type[["ceiling"]][["part"]], drop = TRUE)
      
      #Material
      if ("material" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["material"]])){
          replacement = l_part[["material"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["room"]], 
            model_adapted[["data_table"]][["room"]], replacement, "ceiling_material", "ceiling_elementary_component",
            coating = "ceiling_coating")
        }
      }
      
      #Insulating
      if ("insulating" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["insulating"]])){
          replacement = l_part[["insulating"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["room"]], 
            model_adapted[["data_table"]][["room"]], replacement, "ceiling_insulating", 
            "ceiling_insulating_elementary_component")
        }
      }
    }
  
  #Coating
    if ("coating" %in% names(l_type) | "wall" %in% names(l_type)){ #also need to calculate the coating on a new changed wall
      measure_cost = coating_adaptation_cost(measure_cost, model, model_adapted)
    }  
     
  #Floor
    if ("floor" %in% names(l_type)){
      l_part = split(l_type[["floor"]], l_type[["floor"]][["part"]], drop = TRUE)
      
      #Material
      if ("material" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["material"]])){
          replacement = l_part[["material"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["room"]], 
            model_adapted[["data_table"]][["room"]], replacement, "floor_material", "floor_elementary_component",
            coating = "floor_coating", insulating = "floor_insulating")
        }
      }
      
      #Coating
      if ("coating" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["coating"]])){
          replacement = l_part[["coating"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["room"]], 
            model_adapted[["data_table"]][["room"]], replacement, "floor_coating", "floor_coating_elementary_component")
        }
      }
    }
    
  #Opening
    if ("opening" %in% names(l_type)){
    l_part = split(l_type[["opening"]], l_type[["opening"]][["part"]], drop = TRUE)
      
      #Material
      if ("material" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["material"]])){
          replacement = l_part[["material"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["opening"]], 
            model_adapted[["data_table"]][["opening"]], replacement, "material", "opening_elementary_component",
            sub_type = "opening_type")
        }
      }
    
      #Shutter
      if ("shutter" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["shutter"]])){
          replacement = l_part[["shutter"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["opening"]], 
            model_adapted[["data_table"]][["opening"]], replacement, "shutter", "shutter_elementary_component",
            sub_type = "opening_type")
        }
      }
    }
    
    #Wall
    if ("wall" %in% names(l_type)){
      l_part = split(l_type[["wall"]], l_type[["wall"]][["part"]], drop = TRUE)
      
      #Material
      if ("material" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["material"]])){
          replacement = l_part[["material"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["wall"]], 
            model_adapted[["data_table"]][["wall"]], replacement, "material", "wall_elementary_component", 
            sub_type = "wall_type", render = "render")
        }
      }
      
      #Insulating
      if ("insulating" %in% names(l_part)) {
        for (i in 1:nrow(l_part[["insulating"]])){
          replacement = l_part[["insulating"]][i,]
          measure_cost = elementary_adaptation_cost(measure_cost, material_costs, model[["data_table"]][["wall"]], 
            model_adapted[["data_table"]][["wall"]], replacement, "insulating", "insulating_elementary_component")
        }
      }
    }
    
  }
  
  
  
  
###Return the value of the total cost of the measures
  #Save total values
  ind = which(measure_cost[["detail"]][["measure_name"]] == "total")
  measure_cost[["detail"]][ind, "prevention_cost"] = measure_cost[["prevention"]]
  measure_cost[["detail"]][ind, "repair_cost"] = measure_cost[["repair"]]
  measure_cost[["detail"]][ind, "crawlspace_cost"] = measure_cost[["crawlspace"]]
  measure_cost[["detail"]][ind, "slab_cost"] = measure_cost[["slab"]]
  
  
  return(measure_cost)
}
