#' Calculate the function giving surface depending on height
#' 
#' @param wall data.frame of wall (or coating)
#' @param opening data.frame of opening, possibly NULL
#' @param version_building a character giving which version of specific shall be used (depends on version in floodam.building::BUILDING). Default to "version_03"
#' @return a function giving surface depending on height (absolute height)
#' 
#' @keywords internal

calculate_surface_height = function(wall, opening, version_building = "version_03") {
	if (version_building == "version_03") return(calculate_surface_height_03(wall, opening))
	if (version_building <= "version_02") return(calculate_surface_height_02(wall, opening))
	stop(sprintf("No calculate_surface_height implementation available for version %s of building", version_building))
}

calculate_surface_height_03 = function(
		wall,
		opening,
		talkative = getOption("floodam_building_talkative")
	)
{	
	x = seq(wall[["H_abs"]], wall[["H_abs"]] + wall[["height"]], by = 1) # In centimeter
	y = rep(wall[["width"]], length(x)) # In meter

	if (!is.null(opening)) {
		opening = opening[
			opening[["building"]] == as.character(wall[["building"]]) & 
			opening[["storey"]] == as.character(wall[["storey"]]) & 
			opening[["room"]] == as.character(wall[["room"]]) & 
			opening[["wall"]] == as.character(wall[["wall"]]), ]
		
		if (nrow(opening) > 0) {
			for (i in seq(nrow(opening))) {
				selection = x >=  opening[i, "H_abs"] & x <= opening[i, "H_abs"] + opening[i, "height"]
				y[selection] = y[selection] - opening[i, "width"] / 100
			}
		}
	}

	return(stats::approxfun(x / 100, c(0, cumsum(y)[-length(y)] / 100), method = "linear", rule = 1:2))
}

calculate_surface_height_02 = function(
		wall,
		opening,
		talkative = getOption("floodam_building_talkative")
	)
{
	if (talkative) {
		message(sprintf("\t\t- Calculating surface_height for %s...\n", format_component(wall, "type")))
	}

	x = seq(wall[["H_abs"]], wall[["H_abs"]] + wall[["height"]], by = 1) # In centimeter
	y = rep(wall[["width"]], length(x)) # In meter

	if (!is.null(opening)) {
		opening = opening[opening[["storey"]] == as.character(wall[["storey"]]) & 
			opening[["room"]] == as.character(wall[["room"]]) & 
			opening[["wall"]] == as.character(wall[["wall"]]), ]
		
		if (nrow(opening) > 0) {
			for (i in seq(nrow(opening))) {
				selection = x >=  opening[i, "H_abs"] & x <= opening[i, "H_abs"] + opening[i, "height"]
				y[selection] = y[selection] - opening[i, "width"] / 100
			}
		}
	}

	return(stats::approxfun(x / 100, c(0, cumsum(y)[-length(y)] / 100), method = "linear", rule = 1:2))
}
