#' Adjust elementary damaging function to local characteristics
#' 
#' @param component data.frame with one row, useful information (H_abs, dilapidation, quantity, height)
#' @param what character, what should be used in component
#' @param damaging, array of class elementary_damaging
#' @param dilapidation, numeric, dilapidation rate to be applied, default to 0
#' @param surface_height function, how to calculate surface depending on H_abs, only valid for elementary components linked to wall, default to NULL
#' @param hazard_range list, desired hazard_range, default to getOption("floodam_building_hazard_range")
#' @param synthetic, boolean, should result be synthetic
#' @param graph, boolean, should result be plotted
#' @param talkative, boolean, should the function be talkative (for debug purpose)
#' @param file_log, character, path of the file where to write some possible problems
#' @return an array of two dimensions with same h_range and d_range than damaging
#' 
#' @keywords internal


adjust_elementary_damaging = function(
		component,
		what = NULL,
		damaging = NULL,
		dilapidation = 0,	
		surface_height = NULL,
		hazard_range = NULL,
		synthetic = TRUE,
		graph = FALSE,
		talkative = getOption("floodam_building_talkative"),
		file_log
) {

	if (talkative) {
		message(sprintf("\t\t- Adjusting elementary damage function for %s...\n", format_component(component, what)))
	}

	# Dealing with common errors
	if (is.null(what)) {stop("Need to know which type of elementary component is processed!")}
	if (nrow(component) != 1) {stop("Need to have one and only one component")}

	id = component[[what]]
	if (is.na(id)) return(NULL)
	# if (is.null(damaging)) {stop("Need to have a damaging function!")}
	if (is.null(damaging)) {return(NULL)}


	if ("surface" %in% attributes(damaging)[["support"]] & (
		is.null(surface_height) & is.null(component[["surface"]]))) {
		stop(sprintf("Problem with component %s. Information on surface is needed!", id))
	}
	if (length(grep("h_alea", attributes(damaging)[["h_action"]])) > 0 &
		!is.function(surface_height)) {
		stop(sprintf("Problem with component %s. A function surface_height is needed!", id))
	}	
	if (length(grep("h_alea", attributes(damaging)[["h_action"]])) > 0 &
		is.null(component[["height"]])) {
		stop(sprintf("Problem with component %s. An height is needed!", id))
	}	
	if (is.null(hazard_range)) {
		hazard_range = list(
			h = as.integer(dimnames(damaging)[["h"]]),
			d = as.integer(dimnames(damaging)[["d"]])
		)
	}

	h_range = as.integer(dimnames(damaging)[[1]])
	selection = in_range(hazard_range[["h"]], h_range, "ee")
	if (any(!(as.character(selection) %in% row.names(damaging)))) {
		resolution_asked = unique(diff(hazard_range[["h"]]))
		resolution_available = unique(diff(h_range))
		msg = c(
			sprintf("Resolution for hazard_range (%s) and damaging (%s) not compatible. ",
				unique(diff(hazard_range[["h"]])),
				unique(diff(h_range))
			),
			"You should either:\n",
			"\t- change hazard_range resolution\n",
			"\t- change DAM (be careful it is a bad declared global variable) with desired resolution\n"
		)
		stop(msg)
	}

	## Possibily missing informations for error or log messages
	storey = if (is.null(component[["storey"]])) "---" else as.character(component[["storey"]])
	room = if (is.null(component[["room"]])) "---" else as.character(component[["room"]])
	wall = if (is.null(component[["wall"]])) "---" else as.character(component[["wall"]])

	H_abs = if (is.null(component[["H_abs"]])) 0 else component[["H_abs"]]

	if (!is.null(component[["dilapidation"]]) & dilapidation > 0) {
		warnings("Hum. Two sources of dilapidation are provided! General one not included!")
		dilapidation = 0
	}
	dilapidation_component = if (is.null(component[["dilapidation"]])) 0 else component[["dilapidation"]]

	quantity = if (is.null(component[["quantity"]])) 1 else component[["quantity"]]
	v_component = if (all(c("v_min", "v_max") %in% names(component))) mean(unlist(component[c("v_min", "v_max")])) else NA
	v_damaging = mean(unlist(attributes(damaging)[["v"]]))
	
	# Creating empty damaging function according to hazard_range, with characteristics of damaging
	result = array(as.numeric(NA),
		dim = c(length(hazard_range[[1]]), length(hazard_range[[2]]), dim(damaging)[3]),
		dimnames = c(hazard_range, dimnames(damaging)[3]))
	class(result) = "elementary_damaging"
	attr(result, "h_action") = attr(damaging, "h_action")
	attr(result, "dilapidation") = attr(damaging, "dilapidation")
	attr(result, "support") = attr(damaging, "support")
	attr(result, "value") = attr(damaging, "value")
	attr(result, "component_elementary") = attr(damaging, "component_elementary")
	attr(result, "detail") = component
	attr(result, "dilapidation_general") = dilapidation
	attr(result, "dilapidation_component") = dilapidation_component

	# Managing H_abs: 1. shift values
	selection = in_range(hazard_range[["h"]] - H_abs, h_range, "ee")
	if (any(!(as.character(selection) %in% row.names(damaging)))) {
		h_abs = round(H_abs, -1)
		selection = in_range(hazard_range[["h"]] - h_abs, h_range, "ee")
		if (any(!(as.character(selection) %in% row.names(damaging)))) {
			stop(sprintf("Problem with component %s (storey:%s, room:%s, wall:%s). Apparently H_abs (%s) not compatible with resolution in DAM.", 
				id, storey, room, wall, H_abs))
		} else {
			report = sprintf("Problem with component %s (storey:%s, room:%s, wall:%s). Apparently H_abs (%s) not compatible with resolution in DAM, changed to %s",
				id, storey, room, wall, H_abs, h_abs)
			write(report, file = file_log, append = TRUE)
			warning(sprintf("Problem with some H_abs. See %s for details", file_log))
			H_abs = h_abs
		}
	}
	result[as.character(selection + H_abs),,] =	damaging[as.character(selection),,]
	
	# Managing H_abs: 2. Dealing with value of h greater than what is available in damaging
	n = length(as.character(hazard_range[["h"]][hazard_range[["h"]] > max(h_range + H_abs)]))
	result[as.character(hazard_range[["h"]][hazard_range[["h"]] > max(h_range + H_abs)]), , ] =
		damaging[rep(as.character(max(h_range)), n), , ]
		
	# Managing H_abs: 3. No damage for h = 0
	result[as.character(0), , ] = NA
	
	# Adjusting damaging with quantity, surface and dilapidation
	for (a in dimnames(damaging)[["action"]]) {
		surface = switch(attributes(damaging)[["support"]][a],
			"surface" = if (is.null(component[["surface"]])) surface_height(component[["height"]] / 100) else component[["surface"]],
			"unit" = 1,
			stop(sprintf("Support %s not yet treated! (%s)", attributes(damaging)[["support"]][a], id)))
		
		h_action = grep("h_alea", attributes(damaging)[["h_action"]][a], value = TRUE)
		if (length(h_action) == 1) {
			h_action = gsub("h_alea[+]*", "", h_action)
			h_action = if (h_action == "") 0 else hazard_range[["h"]][which.min(abs(hazard_range[["h"]] - as.integer(h_action)))]
			surface = surface_height((hazard_range[["h"]] + h_action) / 100)
		}
		
		v_correction = 1
		if (attributes(damaging)[["value"]][a] == "value" & !is.na(v_component) & !is.na(v_damaging)) {
			v_correction = v_component / v_damaging
		}
		
		result[ , ,a] = result[ , ,a] * v_correction * (1 - attributes(damaging)[["dilapidation"]][a] * dilapidation_component / 100) * surface * quantity
	}
	result = result * (1 - dilapidation / 100) # if dilapidation_component > 0 the dilapidation = 0 (see below)

	if (!is.null(component[["value"]])) if (component[["value"]] > 0) result[result > component[["value"]]] = component[["value"]] # Necessary for furniture component.
	
	if (graph) {plot(result, d_select = c(12, 72, 120))}
	if (synthetic) {invisible(summarize_elementary_damaging(result))}
	invisible(result)
}

format_component = function(component, what) {
	building = if (is.null(component[["building"]])) "---" else as.character(component[["building"]])
	storey = if (is.null(component[["storey"]])) "---" else as.character(component[["storey"]])
	room = if (is.null(component[["room"]])) "---" else as.character(component[["room"]])
	wall = if (is.null(component[["wall"]])) "---" else as.character(component[["wall"]])
	id = if (is.null(what)) "NULL" else if (is.null(component[[what]])) "---" else as.character(component[[what]])
	what = if (is.null(what)) "NULL" else what
	return (sprintf("%s[%s] in %s/%s/%s/%s", what, id, building, storey, room, wall))
}
