test_that("Test hydraulic_functionalities",
    {
        # set-up
        model_path = list(
            data = system.file("extdata", package = "floodam.building"),
            output = tempdir()
        )
        model_simple = analyse_model(
            model = "adu_simple",
            type = "adu",
            stage = c("load", "extract", "damaging", "hydraulic"),
            verbose = FALSE,
            path = model_path
        )

        flood_simple = expect_no_message(
            generate_limnigraph(
                model = model_simple,
                time = c(0, 54, 108),
                depth = cbind(
                    facade_1 = c(0, 2, 0)
                ),
                exposition = list(
                    facade_1 = list(external = c(
                        "wall_A", "wall_B", "wall_C", "wall_D")
                )),
                verbose = FALSE
            )
        )

        # tests on all functionalities
        # scenario: open, clearance: c(0.3, 0.3), height_break: 1.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "open",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 1.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

         # scenario: open, clearance: c(0.3, 0.3), height_break: 0.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "open",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 0.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: close, clearance: c(0.3, 0.3), height_break: 1.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "close",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 1.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: close, clearance: c(0.3, 0.3), height_break: 0.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "close",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 0.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: combine, clearance: c(0.3, 0.3), height_break: 1.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "combine",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 1.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: combine, clearance: c(0.3, 0.3), height_break: 0.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "combine",
                clearance = c(under = 0.3, side = 0.3),
                height_break = 0.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: close, clearance: c(0.3, 0.3), height_break: Inf
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "close",
                clearance = c(under = 0.3, side = 0.3),
                height_break = Inf,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # scenario: close, clearance: c(0.3, 0.3), height_break: 1.0, clearance: 0.0
        hydraulic = expect_no_error(
            analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "close",
                clearance = c(under = 0.0, side = 0.0),
                height_break = 1.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	        )
        )

        # test equal with flooding under ceiling and no break
        ref_under = matrix(c(0.02138747, 2, 0.02138747, 107, 54, 107),
            nrow = 2,
            byrow = TRUE, 
            dimnames = list(c("z", "t"), c("room_1", "facade_1", "mean")))
        hydraulic = analyse_hydraulic(
                model_simple,
                flood_simple,
                opening_scenario = "close",
                clearance = c(under = 0.2, side = 0.2),
                height_break = Inf,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	    )
        expect_equal(ref_under, hydraulic[["hmax"]])

        flood_simple_over = expect_no_message(
            generate_limnigraph(
                model = model_simple,
                time = c(0, 54, 108),
                depth = cbind(
                    facade_1 = c(0, 4, 0)
                ),
                exposition = list(
                    facade_1 = list(external = c(
                        "wall_A", "wall_B", "wall_C", "wall_D")
                )),
                verbose = FALSE
            )
        )

        # test equal with flooding over ceiling
        ref_over = matrix(c(3, 4, 3, 45, 54, 45), nrow = 2, byrow = TRUE,
            dimnames = list(c("z", "t"), c("room_1", "facade_1", "mean")))
        hydraulic = analyse_hydraulic(
                model_simple,
                flood_simple_over,
                opening_scenario = "close",
                clearance = c(under = 0.2, side = 0.2),
                height_break = 1.0,
                stage = c("hydraulic", "damaging"),
                verbose = FALSE
	    )
        expect_equal(ref_over, hydraulic[["hmax"]])
    }
)
