#' Update damaging function for a given APE code
#' 
#' @param category name of the category to be updated
#' @param statistic data.frame of useful statistic for extrapolation
#' @param path list of useful path
#' @param verbose a boolean indicating if the function should be verbose
#' @param graph a boolean indicating if a graph should be plotted
#' @return Nothing
#' 
#' @keywords internal

update_ape_damaging = function(
		category,
		statistic,
		path = getOption("floodam_building_path"),
		verbose = FALSE,
		graph = TRUE
	)
{
	
	start = Sys.time()
	if (verbose) message(sprintf("Updating damaging for category %s...\n", category))
	
	damaging = utils::read.csv2(file.path(path[["category_output"]], category, "damaging.csv"))
	dim_damaging = list(h = unique(damaging[["h"]]), d = unique(damaging[["d"]]))
	
	for (i in grep(sprintf("^%s$", category), statistic[["category"]])) {
		path_ape = file.path(path[["extrapolation_output"]], statistic[i, "APE_05"])
		dir.create(path_ape, showWarnings = FALSE, recursive = TRUE)

		result = damaging[c("h", "d")]
		result[["surface_building"]] = damaging[["surface_building"]]
		result[["employee_building"]] = 
			if (statistic[i, "employee"] == 0) {
				as.numeric(NA) 
			} else {
				damaging[["relative_building"]] * (statistic[i, "e_emp"] + statistic[i, "s_emp"])
			}
		result[["employee_equipment"]] = 
			if (statistic[i, "employee"] == 0) {
				as.numeric(NA)
			} else {
				damaging[["relative_equipment"]] * statistic[i, "e_emp"]
			}
		result[["employee_stock"]] = 
			if (statistic[i, "employee"] == 0) {
				as.numeric(NA) 
			} else {
				damaging[["relative_stock"]] * statistic[i, "s_emp"]
			}
		
		temp = list(
			surface = 
				matrix(
					result[["surface_building"]],
					nrow = length(dim_damaging[["h"]]),
					ncol = length(dim_damaging[["d"]]),
					dimnames = dim_damaging
				),
			building = 
				matrix(
					result[["employee_building"]],
					nrow = length(dim_damaging[["h"]]),
					ncol = length(dim_damaging[["d"]]),
					dimnames = dim_damaging
				),
			equipment = 
				matrix(
					result[["employee_equipment"]],
					nrow = length(dim_damaging[["h"]]),
					ncol = length(dim_damaging[["d"]]),
					dimnames = dim_damaging
				),
			stock = 
				matrix(
					result[["employee_stock"]],
					nrow = length(dim_damaging[["h"]]),
					ncol = length(dim_damaging[["d"]]),
					dimnames = dim_damaging
				)
			)

		# Check if results are increasing
		fun_test = function(x)	{
			any(apply(x, 1, function(x) {any(diff(x) < 0)}), na.rm = TRUE) | 
			any(apply(x, 2, function(x) {any(diff(x) < 0)}), na.rm = TRUE)
		}
		result_test = sapply(temp, fun_test)
		if (any(result_test)) {
			msg = sprintf(
				"%s of %s not increasing (categoory : %s)",
				paste(names(result_test[result_test]), collapse = ", "),
				statistic[i, "APE_05"],
				category
			)
			warning(msg)
		}
		
		if (graph == TRUE & statistic[i, "employee"] != 0) {			
			temp[["total"]]  = temp[["building"]] + temp[["equipment"]] + temp[["stock"]]
			plot.damaging(temp[-1], path = path_ape, 
				file_name = sprintf("%s-damaging.pdf", statistic[i, "APE_05"]),
				damaging_unit = "euro/emp.")
		}	

		utils::write.csv2(result, file.path(path_ape, "damaging.csv"), row.names = FALSE)
	}
	
	code = statistic[grep(sprintf("^%s$", category), statistic[["category"]]), "APE_05"]
	duration = Sys.time() - start
	if (verbose) {
		message(sprintf("\t... %s APE codes treated (%s)\n", length(code), paste(code, collapse = ", ")))
		message(sprintf("\t... Elapsed time %.2f %s\n", duration, attr(duration, "units")))
	}

	invisible(code)
}
