#' @title Prepare elevator's elementary component input
#' 
#' @description
#' `prepare_elevator_data()` is an **internal** function that creates a database
#' to enable `update_elementary_damaging()` to include elevator elementary
#' function damages.
#' 
#' @param path list of path with at least item damaging (where to find input
#' 		data) and damaging_output (where to save output data)
#' @param na_value character, string to use in column action in case the value
#' 		is NA; default to "NAaction"
#' @param verbose logical, default to getOption("floodam_building_verbose")
#'
#' @return a data.frame
#' 
#' @keywords internal

prepare_elevator_data = function(
		path = getOption("floodam_building_path"),
		na_value = "NAaction",
		verbose = getOption("floodam_building_verbose")
	)
{
	log_file = file.path(path[["damaging_output"]], "damaging.log")
	
	# Loading data
	elevator = utils::read.csv2(
		file.path(
			path[["damaging"]], 
			"elevator",
			"elevator_vulnerability.csv"
		), 
		stringsAsFactors = FALSE
	)
	hypotheses = utils::read.csv2(
		file.path(
			path[["damaging"]], 
			"elevator", 
			"elevator_complementary.csv"
		), 
		stringsAsFactors = FALSE
	)

	# Aggregating minimum and maximum values by elevator type in 'elevator' data
	# using categories in component_elementary column (using 'formula' method 
	# because of its flexibility)
	elementary = stats::aggregate(
		cbind(
			hydraulic_min = v_min * hydraulic, 
			hydraulic_max = v_max * hydraulic,
			traction_min = v_min * traction, 
			traction_max = v_max * traction
		) ~ component_elementary + 
			component_elementary_label_fr + 
			v_unit + 
			v_support, 
		data = elevator, 
		FUN = sum
	)

	# reshaping from 'wide' to 'long' format and tidying the resulting table
	elementary = stats::reshape(
		elementary,
		idvar = "component_elementary",
		varying = list(
			c("hydraulic_min", "traction_min"), 
			c("hydraulic_max", "traction_max")
		),
		times = c("hydraulic", "traction"),
		v.names = c("v_min", "v_max"),
		direction = "long"
	)

	elementary[["component_elementary"]] = sprintf(
		"%s_%s", 
		elementary[["component_elementary"]],
		elementary[["time"]]
	)

	elementary[["component_elementary_label_fr"]] = sprintf(
		"%s_%s", 
		elementary[["component_elementary_label_fr"]],
		elementary[["time"]]
	)

	row.names(elementary) = NULL
	elementary[["time"]] = NULL

	# Since actions do not have probabilities assigned nor flood depths we
	# choose to merge all actions in one using a concatenation of their labels
	# as label. Actions are not the same depending on the elevator type so we
	# will take this into account
	action_hydraulic = stats::aggregate(
		action ~ component_elementary, 
		data = elevator[elevator[["hydraulic"]] == 1, ], 
		FUN = function(x) {
			x[is.na(x)] = na_value
			paste(sort(unique(x)), collapse = "_")
		},
		na.action = stats::na.pass
	)
	action_hydraulic[["component_elementary"]] = sprintf(
		"%s_hydraulic", 
		action_hydraulic[["component_elementary"]]
	)

	action_traction = stats::aggregate(
		action ~ component_elementary, 
		data = elevator[elevator[["traction"]] == 1, ], 
		FUN = function(x) {
			x[is.na(x)] = na_value
			paste(sort(unique(x)), collapse = "_")
		},
		na.action = stats::na.pass
	)
	action_traction[["component_elementary"]] = sprintf(
		"%s_traction", 
		action_traction[["component_elementary"]]
	)

	action = rbind(action_hydraulic,action_traction)

	# Merging elementary components and actions datasets
	result = merge(elementary, action, all.x = TRUE)
	result[["action"]][is.na(result[["action"]])] = na_value

	# merging elementary components and actions and rest of hypotheses datasets
	result = merge(result, hypotheses, all.x = TRUE)
	row.names(result) = result[["component_elementary"]]

	write(
		sprintf(
			"elevator elementary components handled sucessfully on %s.\n", 
			Sys.Date()
		),
		log_file
	)

	return(result)
}
