#' Plot damaging function
#' 
#' @param x array giving damaging function
#' @param d_select selected values of durations where to make some focus. At list one value
#' @param h_range, height range for making the graphs. Default to NULL, and so analyzing dam
#' @param d_range, duration range for making the graphs. Default to NULL, and so analyzing dam
#' @param path path for saving the result. Default to NULL, nothing is saved, graph is displayed
#' @param file_name, name of the elementary component processed. Default to NULL, no title
#' @param damaging_unit, unit of damaging
#' @param option, character, defines what type of damaging will be plotted. Two options: "all" or "total". Default to "all"
#' @param ... a list of extra-parameters (unused)
#' @return NULL
#' 
#' @export

plot.damaging = function(
		x,
		d_select = getOption("floodam_building_d_select"),
		h_range = NULL,
		d_range = NULL, 
		path = NULL,
		file_name = NULL,
		damaging_unit = "euro",
		option = "all",
		...
	)
{
	
	if (!is.null(path)) 
	{
		if (is.null(file_name)) file_name = "damaging.pdf"
		grDevices::pdf(file.path(path, file_name), width = 15 , height = 10)
	} else {
		default_par = graphics::par(no.readonly = TRUE)
	}

	if (option == "total") {
		plot_image_damaging(x[["total"]], h_range, d_range)
		if (!is.null(path)) {
			grDevices::dev.off()
			return(sprintf("Total damaging function saved in %s", file.path(path, file_name)))
		} else {
			graphics::par(default_par)
			return(invisible(NULL))
		}	
	}

	# Prepare 4 different screens for general layout
	graphics::close.screen(all = TRUE)
	mat = rbind(c(0, 1, .9, 1), c(0, 1, .5, .95), c(0, 1, .07, .5), c(0, 1, 0, .1))
	graphics::split.screen(mat)
	  
	graphics::screen(1)
	if (!is.null(file_name)) {
		graphics::title(gsub("[_]", " ", strsplit(file_name, "-damaging")[[1]][1]), cex.main = 1.5)
	}

	# Deal with general graphic, prepare 3 subscreens in screen 2
	mat = rbind(c(0, .45, 0, 1), c(0.45, .5, 0, 1), c(.5, 1, 0, 1))
	graphics::split.screen(mat, screen = 2)

	# Plot the total damage as an image
	graphics::screen(5)	
	plot_image_damaging(x[["total"]], h_range, d_range)

	# Plot the total damage, one line by duration
	graphics::screen(7)
	plot_line_damaging(x[["total"]], d_select, damaging_unit)
	graphics::close.screen(5:7)
	
	# Plot damage by component, one plot by duration
	graphics::split.screen(c(1, length(d_select)), screen = 3)
	col = c("total" = "black", "building" = "blue", "equipment" = "red", "stock" = "green", "furniture" = "red")
  
	for (i in seq(length(d_select))) {
		graphics::screen(i + 4)
		plot_decomposition_line_damaging(x, d_select[i])
	}
	
	# Plot legend of damage by actions, one legend for all
	graphics::screen(4)
	graphics::legend("center", legend = names(x), lwd = 1, col = col[names(x)], horiz = TRUE)
	
	graphics::close.screen(all = TRUE)

	if (!is.null(path)) {
		grDevices::dev.off()
		return(sprintf("Damaging function saved in %s", file.path(path, file_name)))
	} else {
		graphics::par(default_par)
		return(invisible(NULL))
	}	
}

plot_image_damaging = function(
		damaging,
		h_range = NULL,
		d_range = NULL,
		path = NULL,
		file_name = NULL
	)
{
	
	if (!is.null(path)) 
	{
		if (is.null(file_name)) file_name = "damaging-image.pdf"
		grDevices::pdf(file.path(path, file_name), width = 10 , height = 7)
	}

	xlim = if (is.null(h_range)) range(as.numeric(rownames(damaging))) else h_range
	ylim = if (is.null(d_range)) range(as.numeric(colnames(damaging))) else d_range
	suppressWarnings(
		fields::image.plot(
			x = as.integer(rownames(damaging)), 
			y = as.integer(colnames(damaging)), 
			z = damaging, 
			col = grDevices::colorRampPalette(c("blue", "red"))(10),  #gray((10:0)/10), 
			xlab = "h (cm)", ylab = "d (hours)",
			xlim = xlim, ylim = ylim,
			zlim = c(0, max(1, max(damaging, na.rm = TRUE))),
			main = "Full space definition"
		)
	)
	
	if (!is.null(path)) {
		grDevices::dev.off()
		return(sprintf("Image view of damaging function saved in %s", file.path(path, file_name)))
	}
}

plot_line_damaging = function(
		damaging,
		d_select = getOption("floodam_building_d_select"),
		damaging_unit = "euro",
		path = NULL,
		file_name = NULL
	)
{

	if (!is.null(path)) 
	{
		if (is.null(file_name)) file_name = "damaging-line.pdf"
		grDevices::pdf(file.path(path, file_name), width = 10 , height = 7)
	}

	plot(
		NA,
		xlim = range(as.integer(rownames(damaging))), 
		ylim = range(0, max(1, apply(damaging, 1:2, sum, na.rm = TRUE))), 
		xlab = "h (cm)", ylab = sprintf("damage (%s)", damaging_unit),
		main = "Total"
	)
	col = grDevices::rainbow(length(d_select))
	lapply(as.character(d_select), 
	function(z) {
		graphics::lines(as.integer(rownames(damaging)), damaging[ , z], 
			type = "l", col = col[which(d_select == z)])
	}
	)
	graphics::legend("bottomright", title = "Duration", legend = paste(d_select, "hours"), lwd = 1, seg.len = .5, col = col)

	if (!is.null(path)) {
		grDevices::dev.off()
		return(sprintf("Line view of damaging function saved in %s", file.path(path, file_name)))
	}
}

plot_decomposition_line_damaging = function(
		damaging,
		d_select = getOption("floodam_building_d_select")[1],
		col = NULL,
		damaging_unit = "euro",
		add_legend = FALSE,
		path = NULL,
		file_name = NULL
	)
{
	
	if (!is.null(path)) 
	{
		if (is.null(file_name)) file_name = sprintf("damaging-decomposition-line-%s.pdf", d_select)
		grDevices::pdf(file.path(path, file_name), width = 10 , height = 7)
	}
	if (is.null(col)) {
		col = c("total" = "black", "building" = "blue", "equipment" = "red", "stock" = "green", "furniture" = "red")
	}
	
	plot(NA,
		xlim = range(as.integer(rownames(damaging[["total"]]))), 
		ylim = range(damaging[["total"]]), 
		xlab = "h (cm)", ylab = sprintf("damage (%s)", damaging_unit),
		main = sprintf("Duration: %s hours", d_select)
	)
	
	for (a in names(damaging)) {
		graphics::lines(as.integer(rownames(damaging[["total"]])), damaging[[a]][ , as.character(d_select)], type = "l", col = col[a])
	}

	if (add_legend == TRUE) 	graphics::legend("topleft", legend = names(damaging), lwd = 1, col = col[names(damaging)])

	if (!is.null(path)) {
		grDevices::dev.off()
		return(sprintf("Line view of decomposed damaging function saved in %s", file.path(path, file_name)))
	}
}
