#' plot a top perspective of a building
#' 
#' @param data_table, list of needed tables
#' @param model_path, list of needed paths
#' @param device, character, define way of plotting. Three options: "display", "pdf" or "png". Default to "display".
#' @param legend_table, data.frame, default to floodam.building::LEGEND
#' @param finition, character, define type of finishing for plotting. Two options: "check" or "presentation". Default to "check".
#' @param selection, character, selection of building to be plotted
#' @param what_floor, character, define what type of component is used to plot floor. Default to "floor_elementary_component".
#' @return A message
#' 
#' @keywords internal

plot_building_top_03 = function(
		data_table,
		model_path = NULL,
		device = "display",
		legend_table = floodam.building::LEGEND,
		finition = "check",
		selection = NULL,
		what_floor = "floor_elementary_component"
	)
{
	top_by_opening = function(opening, wall, col) {
		# Definition of points
		point = center = c(opening[["x"]], opening[["y"]])
		wall_a = c(wall[["x"]], wall[["y"]])
		wall_b = c(wall[["xx"]], wall[["yy"]])

		#	Projection of center on wall
		norm = (wall_b - wall_a) / sqrt(sum((wall_b - wall_a)^2))
		scalar = sum((center - wall_a) * norm)
		center = wall_a + sum((center - wall_a) * norm) * norm

		# Creation of opening (rectangle) around center
		initial = data.frame(
			x = opening[["width"]] / 200 * c(-1, 1, 1, -1)  + center[1], # opening width in cm
			y = 20 / 200 * c(-1, -1, 1, 1) + center[2])        			# opening thickness of 10 cm
		
		# Rotation of opening
		final = data.frame(
			x = center[1] + (initial[["x"]] - center[1]) * norm[1] - (initial[["y"]] - center[2]) * norm[2],
			y = center[2] + (initial[["x"]] - center[1]) * norm[2] + (initial[["y"]] - center[2]) * norm[1])
		
		#x' = cos(theta)*(x-xc) - sin(theta)*(y-yc) + xc
    #y' = sin(theta)*(x-xc) + cos(theta)*(y-yc) + yc
		
		# Plot
		graphics::polygon(final[["x"]], final[["y"]], col = col)
		graphics::points(point[1], point[2], pch=".", cex = 5)
	}

	top_by_wall = function(wall, opening) {
		wall_name = levels(droplevels(wall[["wall"]]))
		wall_col = legend_table[as.character(wall[["wall_type"]]), "col"]
		wall_lwd = legend_table[as.character(wall[["wall_type"]]), "lwd"]
		
		wall_col[is.na(wall_col)] = "red"
		wall_lwd[is.na(wall_lwd)] = 2
		
		graphics::lines(wall[, c("x", "xx")], wall[, c("y", "yy")], type = "l", lwd = wall_lwd, col = wall_col)
		
		if (!is.null(opening) && nrow(opening) > 0) {
			opening_col = legend_table[as.character(opening[["opening_type"]]), "col"]
			
			#If one opening does not have coordinates, well treat them all without coordinates
			if (any(is.na(opening[c("x","y")]))) opening[c("x","y")] = create_opening_coordinates(wall[c("x", "xx", "y", "yy")], nrow(opening))

			mapply(top_by_opening, split(opening, seq(nrow(opening))), col = opening_col, MoreArgs = list(wa = wall[c("x", "xx", "y", "yy")]))
		}
	}
	
	top_by_room = function(room, wall, opening) {
		room_name = if (finition == "presentation") as.character(room[["name"]]) else as.character(room[["room"]])
		
		if (length(grep("^external", room[["room"]])) == 1) {
			room_name =  NA
			color = "gray95" 
			density = NULL
			lwd = NULL 
		} else if (is.na(room[[what_floor]])) {
			color = NA 
			density = NULL
			lwd = NULL 
		} else {
			color = legend_table[as.character(room[[what_floor]]), "col"]
			density = legend_table[as.character(room[[what_floor]]), "density"]
			lwd = legend_table[as.character(room[[what_floor]]), "lwd"]
			if (is.na(density) | is.na(lwd)) {
				msg = sprintf("Problem with %s in room %s. Check compatibility with legend_table.", room[[what_floor]], room[["room"]])
				stop(msg)
			}
		}
		
		graphics::polygon(wall[["x"]], wall[["y"]], col = "white", border = NA)
		graphics::polygon(wall[["x"]], wall[["y"]], col = color, density = density, lwd = lwd, border = NA)
		graphics::text(x = mean(wall[["x"]]), y = mean(wall[["y"]]), labels = room_name)
		
		wall = cbind(wall, xx = c(wall[["x"]][-1], wall[["x"]][1]), yy = c(wall[["y"]][-1], wall[["y"]][1]))		

		level = levels(droplevels(wall[["wall"]]))
		wall = split_special(wall, "wall", level)
		opening = split_special(opening, "wall", level)
		mapply(top_by_wall, wall, opening)
	}
	
  top_by_storey = function(storey, room, wall, opening, limit){	
		storey_name = if (finition == "presentation") as.character(storey[["name"]]) else as.character(storey[["storey"]])
		h = storey[["H_abs"]]

    plot(x = NULL, y = NULL, xlab = "x", ylab = "y", xlim = c(limit[["x_min"]], limit[["x_max"]]), 
         ylim = c(limit[["y_min"]], limit[["y_max"]]), main = paste(storey_name, "\nh =", h))

		level = levels(droplevels(room[["room"]]))
		room = split_special(room, "room", level)
		wall = split_special(wall, "room", level)
		opening = split_special(opening, "room", level)

		mapply(top_by_room, room, wall, opening)
  }
  
  top_by_building = function(building, storey, room, wall, opening){	
		building_name = if (finition == "presentation") as.character(building[["name"]]) else as.character(building[["building"]])

		# determining plot limits and some useful parameters for aspect
		limit = list(
			x_min = min(wall[["x"]]),
			x_max = max(wall[["x"]]),
			y_min = min(wall[["y"]]),
			y_max = max(wall[["y"]]))
		mai = c(1, 1, 1, 1)
		width = diff(range(wall[["x"]])) + sum(mai[c(2, 4)])
		height = diff(range(wall[["y"]])) + sum(mai[c(1, 3)])

		level = levels(droplevels(wall[["storey"]]))
		storey = split_special(storey, "storey", level)
		room = split_special(room, "storey", level)
		wall = split_special(wall, "storey", level)
		opening = split_special(opening, "storey", level)

		if ("display" %in% device) {
			default_par = graphics::par(mfrow = c(length(level), 1))
			mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
			graphics::par(default_par)
		}

		if ("pdf" %in% device) {
			file_name = file.path(model_path[["model_output_building_plan"]], sprintf("plan_top_%s.pdf", building[["building"]]))
			grDevices::pdf(file_name, width = width,	height = height)
			graphics::par(mai = c(1, 1, 1, 1), cex = 2)
			mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
			grDevices::dev.off()    
			return(sprintf("Top view of building plan save in %s", file_name))
		}

		if ("png" %in% device) {
			file_name = file.path(model_path[["model_output_building_plan"]], sprintf("plan_top_%s.png", building[["building"]]))
			grDevices::png(file_name, 
				width = width * 100, 
				height = height * 100 * length(level), 
				units = "px", 
				pointsize = 40)
			graphics::par(mfrow = c(length(level), 1))
			graphics::plot.new()
			mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
			grDevices::dev.off()
			return(sprintf("Top view of building plan save in %s", file_name))
		}
	}

 	building = split_special(data_table[["building"]], "building", levels(data_table[["wall"]][["building"]]))
	storey = split_special(data_table[["storey"]], "building", levels(data_table[["wall"]][["building"]]))
	room = split_special(data_table[["room"]], "building", levels(data_table[["wall"]][["building"]]))
	wall = split_special(data_table[["wall"]], "building", levels(data_table[["wall"]][["building"]]))
	opening = split_special(data_table[["opening"]], "building", levels(data_table[["wall"]][["building"]]))

	if (is.null(selection)) selection = seq_along(building)
 	if (all(device == "display")) selection = selection[1]
 	mapply(top_by_building, building[selection], storey[selection], room[selection], wall[selection], opening[selection])
}	

plot_building_top_02 = function(
		data_table,
		model_path = NULL,
		device = "display",
		legend_table = floodam.building::LEGEND,
		finition = "check",
		what_floor = "floor_elementary_component"
	)
{	
	top_by_opening = function(opening, wall, col) {
		# Definition of points
		point = center = c(opening[["x"]], opening[["y"]])
		wall_a = c(wall[["x"]], wall[["y"]])
		wall_b = c(wall[["xx"]], wall[["yy"]])

		#	Projection of center on wall
		norm = (wall_b - wall_a) / sqrt(sum((wall_b - wall_a)^2))
		scalar = sum((center - wall_a) * norm)
		center = wall_a + sum((center - wall_a) * norm) * norm

		# Creation of opening (rectangle) around center
		initial = data.frame(
			x = opening[["width"]] / 200 * c(-1, 1, 1, -1)  + center[1], # opening width in cm
			y = 20 / 200 * c(-1, -1, 1, 1) + center[2])        			# opening thickness of 10 cm
		
		# Rotation of opening
		final = data.frame(
			x = center[1] + (initial[["x"]] - center[1]) * norm[1] - (initial[["y"]] - center[2]) * norm[2],
			y = center[2] + (initial[["x"]] - center[1]) * norm[2] + (initial[["y"]] - center[2]) * norm[1])
		
		#x' = cos(theta)*(x-xc) - sin(theta)*(y-yc) + xc
    #y' = sin(theta)*(x-xc) + cos(theta)*(y-yc) + yc
		
		# Plot
		graphics::polygon(final[["x"]], final[["y"]], col = col)
		graphics::points(point[1], point[2], pch=".", cex = 5)
	}

	top_by_wall = function(wall, opening) {
		wall_name = levels(droplevels(wall[["wall"]]))
		wall_col = legend_table[as.character(wall[["wall_type"]]), "col"]
		wall_lwd = legend_table[as.character(wall[["wall_type"]]), "lwd"]
		
		wall_col[is.na(wall_col)] = "red"
		wall_lwd[is.na(wall_lwd)] = 2
		
		graphics::lines(wall[, c("x", "xx")], wall[, c("y", "yy")], type = "l", lwd = wall_lwd, col = wall_col)
		
		if (!is.null(opening) && nrow(opening) > 0) {
			opening_col = legend_table[as.character(opening[["opening_type"]]), "col"]
			
			#If one opening does not have coordinates, well treat them all without coordinates
			if (any(is.na(opening[c("x","y")]))) opening[c("x","y")] = create_opening_coordinates(wall[c("x", "xx", "y", "yy")], nrow(opening))

			mapply(top_by_opening, split(opening, seq(nrow(opening))), col = opening_col, MoreArgs = list(wa = wall[c("x", "xx", "y", "yy")]))
		}
	}
	
	top_by_room = function(room, wall, opening) {
		room_name = if (finition == "presentation") as.character(room[["name"]]) else as.character(room[["room"]])
		
		if (length(grep("^external", room[["room"]])) == 1) {
			room_name =  NA
			color = "gray95" 
			density = NULL
			lwd = NULL 
		} else if (is.na(room[[what_floor]])) {
			color = NA 
			density = NULL
			lwd = NULL 
		} else {
			color = legend_table[as.character(room[[what_floor]]), "col"]
			density = legend_table[as.character(room[[what_floor]]), "density"]
			lwd = legend_table[as.character(room[[what_floor]]), "lwd"]
			if (is.na(density) | is.na(lwd)) {
				msg = sprintf("Problem with %s in room %s. Check compatibility with legend_table.", room[[what_floor]], room[["room"]])
				stop(msg)
			}
		}
		
		graphics::polygon(wall[["x"]], wall[["y"]], col = "white", border = NA)
		graphics::polygon(wall[["x"]], wall[["y"]], col = color, density = density, lwd = lwd, border = NA)
		graphics::text(x = mean(wall[["x"]]), y = mean(wall[["y"]]), labels = room_name)
		
		wall = cbind(wall, xx = c(wall[["x"]][-1], wall[["x"]][1]), yy = c(wall[["y"]][-1], wall[["y"]][1]))		

		level = levels(droplevels(wall[["wall"]]))
		wall = split_special(wall, "wall", level)
		opening = split_special(opening, "wall", level)
		mapply(top_by_wall, wall, opening)
	}
	
	top_by_storey = function(storey, room, wall, opening, limit){	
			storey_name = if (finition == "presentation") as.character(storey[["name"]]) else as.character(storey[["storey"]])
			h = storey[["H_abs"]]
			
		plot(x = NULL, y = NULL, xlab = "x", ylab = "y", xlim = c(limit[["x_min"]], limit[["x_max"]]), 
			ylim = c(limit[["y_min"]], limit[["y_max"]]), main = paste(storey_name, "\nh =", h))

			level = levels(droplevels(room[["room"]]))
			room = split_special(room, "room", level)
			wall = split_special(wall, "room", level)
			opening = split_special(opening, "room", level)
			mapply(top_by_room, room, wall, opening)
	}
  
	## Preparing
	# determining plot limits and some useful parameters for aspect
	limit = list(
			x_min = min(data_table[["wall"]][["x"]]),
			x_max = max(data_table[["wall"]][["x"]]),
			y_min = min(data_table[["wall"]][["y"]]),
			y_max = max(data_table[["wall"]][["y"]]))
	mai = c(1, 1, 1, 1)
	width = diff(range(data_table[["wall"]][["x"]])) + sum(mai[c(2, 4)])
	height = diff(range(data_table[["wall"]][["y"]])) + sum(mai[c(1, 3)])

	storey = split_special(data_table[["storey"]], "storey", levels(data_table[["wall"]][["storey"]]))
	room = split_special(data_table[["room"]], "storey", levels(data_table[["wall"]][["storey"]]))
	wall = split_special(data_table[["wall"]], "storey", levels(data_table[["wall"]][["storey"]]))
	opening = split_special(data_table[["opening"]], "storey", levels(data_table[["wall"]][["storey"]]))
	
	if ("display" %in% device) {
		default_par = graphics::par(mfrow = c(length(levels(data_table[["wall"]][["storey"]])), 1))
		mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
		graphics::par(default_par)
	}

	if ("pdf" %in% device) {
		grDevices::pdf(
			file.path(model_path[["model_output_building_plan"]], "building_top_plan.pdf"), 
			width = width,
			height = height
		)
		graphics::par(mai = c(1, 1, 1, 1), cex = 2)
		mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
		grDevices::dev.off()    
		return(sprintf("Top view of building plan save in %s", file.path(model_path[["model_output_building_plan"]], "building_top_plan.pdf")))
	}

	if ("png" %in% device) {
		grDevices::png(file.path(model_path[["model_output_building_plan"]], "building_top_plan.png"), 
				width = width * 100, 
				height = height * 100 * length(levels(data_table[["wall"]][["storey"]])), 
				units = "px", 
				pointsize = 40)
		graphics::par(mfrow = c(length(levels(data_table[["wall"]][["storey"]])), 1))
		graphics::plot.new()
		mapply(top_by_storey, storey, room, wall, opening, MoreArgs = list(limit = limit))
		grDevices::dev.off()
		return(sprintf("Top view of building plan save in %s", file.path(model_path[["model_output_building_plan"]], "building_top_plan.png")))
	}
}	
