#' Load a model from basic files.
#' 
#' @param model_name character, name of the model
#' @param type character, type of the model
#' @param path list of useful paths, default to getOption("floodam_building_path")
#' @param file_name, list of useful file names, default to getOption("floodam_building_file_name")
#' @param type_building character, type of building file (building or yaml)
#' @param version_building character, version of building to be used
#' @param inspect boolean, shall information from building file be inspected according to pattern? Default to TRUE
#' @param verbose boolean, shall floodam tells what it is doing, default to getOption("floodam_building_verbose")
#' @return An object of class "model" with those characteristics
#' \describe{
#'   \item{name}{character, name of the model}
#'   \item{type}{character, type of the model}
#'   \item{path}{list, useful paths to read and write data}
#'   \item{file_name}{list, useful file names to read and write data}
#'   \item{building}{list, representation of data stored in the building file}
#'   \item{furniture}{data frame of data in furniture.csv file}
#' }
#' @examples
#' 
#' 	model_path = list(
#' 		data = system.file("extdata", package = "floodam.building"),
#' 		output = tempdir()
#' 	)
#' 	load_model(model = "adu_t", type = "adu", path = model_path)
#' 
#' @export

load_model = function(
		model_name,
		type,
		path = getOption("floodam_building_path"),
		file_name = getOption("floodam_building_file_name"),
		type_building = getOption("floodam_building_type_building"),
		version_building = NULL,
		inspect = TRUE,
		verbose = getOption("floodam_building_verbose")
	)
{
	if (methods::is(model_name, "model")) {
		warning("model_name is already a model! Nothing new is loaded.")
		return(model_name)
	}

	if (! methods::is(model_name, "character") & length(model_name) != 1) {
		stop("model_name is not a name of a model (character of length 1).")
	}

	if (! methods::is(type, "character") & length(type) != 1) {
		stop("type is not a name of a type (character of length 1).")
	}

	if (!is.null(version_building)) {
		if (!version_building %in% names(floodam.building::BUILDING)) {
			warning(
				sprintf(
					"version_building %s is not an admissible version for the data format!",
					version_building
				)
			)
		}
	}

	if (verbose) message(sprintf("Loading model '%s'...\n", model_name))

	# Defining components of model
	path = update_path(model_name, type, path)
	file_name = update_file_name(model_name, file_name)

	# Initialization of log file
	init_log(path, file_name)

	# Load information from building file
	if (type_building == "yaml") {
		file_name[["building"]] = gsub("xml$", "yaml", file_name[["building"]])
		yaml = file.path(path[["model"]], model_name, file_name[["building"]])
		building = yaml::read_yaml(yaml)
	} else {
		xml = file.path(path[["model"]], model_name, file_name[["building"]])
		xml = XML::xmlToList(XML::xmlParse(xml))
		building = remove_comment(rapply(xml, to_numeric, how = 'replace'))
	}

	if (is.null(building[["general"]][["version"]])) {
		building[["general"]][["version"]] = version_building
	}
	if (!building[["general"]][["version"]] %in% names(floodam.building::BUILDING)) {
		stop(
			sprintf(
				"%s is not an admissible version for the data format!",
				building[["general"]][["version"]]
			)
		)
	}
	if (is.null(building[["general"]][["type"]])) {
		building[["general"]][["type"]] = type
	}

	category = grep(
		sprintf("category_%s", building[["general"]][["type"]]),
		names(floodam.building::COMPONENT_ELEMENTARY),
		value = TRUE
	)
	if (length(category) == 0) {
		warning(
			sprintf(
				"No category found in floodam.building::COMPONENT_ELEMENTARY for type %s",
				building[["general"]][["type"]]
			)
		)
	}

	if (length(category) > 1) {
		warning(
			sprintf(
				"Two categories found in floodam.building::COMPONENT_ELEMENTARY for type %s.
				Keeping the first one",
				building[["general"]][["type"]]
			)
		)
		category = category[1]
	}
	
	# checking that items in the 'element' slot are uniquely defined
	building[["element"]] = check_element(building[["element"]])

	# Load furniture
	furniture = load_furniture(model_name, path, file_name)

	# Define model
	result = list(
		name = model_name,
		type = type,
		category = category,
		path = path,
		file_name = file_name,
		building = building,
		data_table = list(furniture = furniture)
	)
	class(result) = "model"
	
	# Inspect building attributes of model
	if (inspect == TRUE) {
		inspect_building(
			result,
			pattern = floodam.building::BUILDING[[building[["general"]][["version"]]]],
			verbose = verbose
		)
	}
	if (verbose) message("\t... successful\n")

	invisible(result)
}

to_numeric = function(x, sep = ",", dec = ".") {
    result = strsplit(x, split = sep)[[1]]
    result = gsub(dec, ".", result, fixed = TRUE)
    transformed = suppressWarnings(as.numeric(result))
    if (all(is.na(transformed))) return(result)
    return(transformed)
}

# Deprecated. Changing options warn create trouble with expect_warning from
# testthat
# old_to_numeric = function(xml) {
# 	o = getOption('warn')
# 	options(warn = 2)
# 	result = strsplit(xml, split = ',')[[1]]
# 	try(result <- as.numeric(result), TRUE)
# 	options(warn = o)
# 	return(result)
# }

remove_comment = function(x) {
    if (is.list(x)) {
        if (length(grep("^comment$", names(x)) != 0)) {
            x = x[-grep("^comment$", names(x))]
        }
        x = lapply(x, remove_comment)
    }
    return(x)
}

