#' Load furniture for a given model.
#' 
#' @param model_name the name given to the model used to store log information
#' @param path list of useful paths
#' @param file_name list of useful file_names
#' @return a data.frame of furniture present in the model. If the file furniture.csv does not yet exist, return NULL.
#' 
#' @details
#' Some tries are made on format:
#' 1. First, 'furniture.csv' is supposed to be readable by utils::read.csv2.
#' 2. If not it is supposed to be readable by utils::read.csv.
#' 3. If not, an error is returned. User is asked to manage format on his own.
#' 
#' Some tries are made on encoding:
#' 1. First, 'furniture.csv' is supposed to be encoded in UTF-8.
#' 2. If this try makes some error, another try is made in ISO-8859-15.
#' 3. If this second try does not work, an error is returned. User is asked to manage encoding on his own.
#' 
#' @examples
#' 
#' 	model = "adu_t"
#' 	type = "adu"
#' 
#' 	load_furniture(
#' 		model_name = model, 
#' 		path = update_path(
#' 			model = model, 
#' 			type = type, 
#' 			path = list(
#' 				data = system.file("extdata", package = "floodam.building"), 
#' 				output = tempdir()
#' 			)
#' 		),
#' 		file_name = init_file_name()
#' 	)
#' 
#' @export

load_furniture = function(
		model_name,
		path,
		file_name
	)
{
	furniture_path = file.path(path[["model"]], model_name, file_name[["furniture"]])
	log_path = file.path(path[["model"]], model_name, file_name[["furniture"]])
	
	if (!(file.exists(furniture_path))) return(NULL)

	# Some checks on format!
	if (
		length(
			scan(
				furniture_path,
				what = "",
				nlines = 1,
				sep =";",
				quiet = TRUE
			)
		) != 1
	) {
		read_fun = utils::read.csv2
	} else if (
		length(
			scan(
				furniture_path,
				what = "",
				nlines = 1,
				sep =",",
				quiet = TRUE
			)
		) != 1
	) {
		read_fun = utils::read.csv
	} else {
		stop(
			sprintf(
				"Format trouble with %s. It should be a csv file readable with utils::read.csv2 (or utils::read.csv)",
				furniture_path
			)
		)
	}
	
	result = try(
		read_fun(
			furniture_path,
			stringsAsFactors = FALSE
		),
		silent = TRUE
	)
	if (methods::is(result, "try-error")) {
		result = try(
			read_fun(
				furniture_path,
				stringsAsFactors = FALSE,
				fileEncoding = "ISO-8859-15"
			),
			silent = TRUE
		)
		if (methods::is(result, "try-error")) {
			stop(
				sprintf(
					"Encoding trouble with %s.",
					furniture_path
				)
			)
		}
		report = sprintf(
			"Encoding trouble with %s. Changed to UTF-8",
			furniture_path
		)
		warning(report)
		write(report, file = log_path, append = TRUE)
	}
	
	if (is.null(result[["room"]])) {result[["room"]] = result[["room_name"]]}
	if (is.null(result[["wall"]])) {result[["wall"]] = result[["wall_name"]]}
	if (is.null(result[["H"]])) {result[["H"]] = result[["relative_height"]]}

	return(result)
}


