#' Keep rows order in data.frame when a function is applied to it.
#' 
#' @param df the data.frame that will be changed by the function fun
#' @param fun the function that will be applied
#' @param ... other parameters to fun
#' @section Source of inspiration:
#' A comment from landroni (2017-09-23 at 15:59) on \url{http://stackoverflow.com}
#' @examples
#' 
#' df_1 = data.frame(object = c('A', 'B', 'D', 'F', 'C'), class = c(2, 1, 2, 3, 1))
#' df_2 = data.frame(class = c(1, 2, 3), prob = c(0.5, 0.7, 0.3))
#' merge(df_1, df_2)
#' keep_df_order(df_1, merge, df_2, suffixes = c("_x","_y"))
#' 
#' @export

keep_df_order = function(df, fun, ...) {
  column_order = "_column_order"
  while(column_order %in% names(df)) column_order = paste(column_order, "x", sep = "_")
  df[column_order] = 1:nrow(df) 
  result = fun(df, ...) 
  if (!column_order %in% colnames(result)) stop("Ordering column not preserved by function") 
  result = result[order(result[[column_order]]), ] 
  result[column_order] = NULL 
  result 
} 
