#' Inspect building part of model.
#' 
#' @param model model
#' @param pattern list, against which integrity of building information is tested
#' @param verbose boolean, should floodam explains what it is doing
#' @return NULL. A message is provided and a report is written.
#' 
#' @keywords internal

inspect_building = function(
        model,
        pattern = floodam.building::BUILDING[[getOption("floodam_building_version_building")]],
        verbose = getOption("floodam_building_verbose")
    )
{
    report = construct_report(model[["building"]], pattern, model[["name"]])
    file_log = file.path(model[["path"]][["model_log"]], model[["file_name"]][["log"]])
    
    if (length(report) > 0) {
        write(report, file = file_log, append = TRUE)
        if (verbose) message(paste("\t- ", report, "\n", sep = ""))
        stop(
            sprintf(
                "%s has some troubles. Look at %s for details.",
                model[["file_name"]][["building"]],
                file_log
            )
        )
    } else {
        report = sprintf(
            "\t- Structure of %s of '%s' has been successfully checked\n",
            model[["file_name"]][["building"]],
            model[["name"]]
        )
        write(report, file = file_log, append = TRUE)

        if (verbose) message(report)
    }
    
    return(NULL)
}

extract_element = function(x, status = "ALL"){
    x = x[sapply(x, is.list)]
    if (status %in% "MANDATORY") x = x[sapply(x, function(x){x[["STATUS"]] == status})]
    
    # General behaviour
    result = list(unique = names(x))
    
    # Dealing with element that may have several occurences
    several = unlist(
        sapply(
            x[result[["unique"]]],
            function(x){if (is.null(x[["SEVERAL"]])) FALSE else x[["SEVERAL"]]}
        )
    )
    result[["several"]] = result[["unique"]][several]
    result[["unique"]] = result[["unique"]][!several]
    
    # Dealing with element that may have different names (they all may have several occurences)
    result[["generic"]] = switch(
        status,
        "MANDATORY" = x[["generic"]][[status]],
        unlist(c(x[["generic"]][c("MANDATORY", "OPTIONAL")]), use.names = FALSE)
    )
    if (is.null(result[["generic"]])) result[["generic"]] = character()

    result[["unique"]] = setdiff(result[["unique"]], "generic")
    result[["several"]] = setdiff(result[["several"]], "generic")
    
    return(result)
}

extract_id = function(x, scope = c("LOCAL", "GLOBAL")){
    scope = match.arg(scope)

    if (length(x) == 0) return(character())

    x = x[sapply(x, is.list)]
    if (length(x) == 0) return(character())

    result = x[
        sapply(
            x,
            function(x){
                if (is.null(x[["ID"]])) {
                    FALSE
                } else {
                    x[["ID"]] %in% scope
                }
            }
        )
    ]

    if ("generic" %in% names(result)) {
        result = c(
            names(result),
            unlist(
                result[["generic"]][c("MANDATORY", "OPTIONAL")],
                use.names = FALSE
            )
        )
        result = setdiff(result, "generic")
    } else {
        result = names(result)
    }
    
    if (scope %in% "GLOBAL") {
        result = union(
            result,
            unlist(lapply(x, extract_id, scope), use.names = FALSE)
        )
    }

    if (is.null(result)) result = character()
    return(result)
}
        
check_missing = function(n, p) {
    list(
        unique = if (length(p[["unique"]]) > 0) {
            p[["unique"]][!(p[["unique"]] %in% n)]
            } else {
            p[["unique"]]
            },
        several = if (length(p[["several"]]) > 0) {
            p[["several"]][
                sapply(
                    p[["several"]],
                    function(x) {length(grep(sprintf("^%s", x), n)) == 0}
                )
            ]
        } else {
            p[["several"]]
        },
        generic = if (length(p[["generic"]]) > 0) {
            p[["generic"]][
                sapply(
                    p[["generic"]],
                    function(x) {length(grep(sprintf("^%s", x), n)) == 0}
                )
            ]
        } else {
            p[["generic"]]
        }
    )
}
        
check_present = function(n, p) {
    result = rep(NA, length(n))
    if (length(p[["unique"]]) > 0) {
        result[n %in% p[["unique"]]] = n[n %in% p[["unique"]]]
    }
    for (e in p[["several"]]) {
        selection = grep(e, n)
        if (length(selection) > 0) result[selection] = e
    }
    if (length(p[["generic"]]) > 0) {
        selection = sapply(
            n,
            function(x) {any(!is.na(pmatch(p[["generic"]], x)))}
        )
        if (length(selection) > 0) result[selection] = "generic"
    }
    return(result)
}

#' Return all names recursively of a list
#' 
#' @param x a list
#' @return Character. All names found in x.
#' 
#' @keywords internal

find_all_names = function(x) {
    stats::setNames(
        sort(c(names(x), unlist(lapply(x[is.list(x)], find_all_names)))),
        NULL
    )
}

construct_report = function(x, pattern, level = "", master = TRUE){
    report = c()

    # Managing missing elements
    missing = check_missing(names(x), extract_element(pattern, status = "MANDATORY"))
    report = update_report(
        missing[["unique"]],
        sprintf("Each of following elements are missing in '%s':", level),
        report
    )
    report = update_report(
        c(missing[["several"]], missing[["generic"]]),
        sprintf("At least one element of each following type is missing in '%s':", level),
        report
    )
        
    # Managing present elements, verification if any element is not expected
    present = check_present(names(x), extract_element(pattern))
    report = update_report(
        names(x)[is.na(present)],
        sprintf("Following elements are not expected in '%s':", level),
        report
    )
    
    # Managing present elements
    x = x[!is.na(present)]
    present = present[!is.na(present)]
    
    # Verification of the type
    check_type = sapply(x, typeof) == sapply(pattern[present], function(x){x[["VALUE"]]})
    report = update_report(
        names(x)[!check_type],
        sprintf("Trouble with type in '%s' for the following elements:", level),
        report
    )

    # Verification of local ID
    for(local_id in extract_id(pattern)) {
        element = grep(sprintf("^%s", local_id), names(x), value = TRUE)
        report = update_report(
            unique(element[duplicated(element)]),
            sprintf(
                "Those elements have several occurrences with the same name in '%s' but should be local ID:",
                level
            ),
            report
        )
    }
    
    # Applying recursive inspection if needed
    recursive = sapply(x, typeof) == "list"
    if (any(recursive)) {
        report = c(
            report,
            unlist(
                mapply(
                    construct_report,
                    x[recursive],
                    pattern = pattern[present[recursive]],
                    level = paste(level, names(x)[recursive], sep="/"),
                    master = FALSE
                )
            )
        )
    }

    # Checking GLOBAL ID, but only once
    if (isTRUE(master)) {
        name_all = find_all_names(x)

        for (global in extract_id(pattern, "GLOBAL")) {
            name = grep(sprintf("^%s", global), name_all, value = TRUE)
            report = update_report(
                unique(name[duplicated(name)]),
                sprintf("Some %ss have duplicated name in '%s':", global, level),
                report
            )
        }
    }
    
    report
}
