#' Find the damage total of the model based on the hydraulic simulation
#'
#' @param h_max_by_room vector of doubles, max water level by room
#' @param damage damaging part of an object of class model
#' @param exposition character or list of characters, giving the names of the
#'  walls concerned by external limnigraphs
#' @param threshold_height numeric, height in centimeters above which a damage
#'  is considered
#' @param threshold_duration numeric, maximum duration for the damage function
#' used for interpolation. See details.
#'
#' @details
#' 
#' The damage functions used for interpolation are those with the smallest
#' positive duration available in `damage`. This minimum must be less than or
#' equal to `threshold_duration`. If this is not the case, an error is returned
#' to force the user to construct the damage functions with `analyse_model()`
#' using a `hazard_range` consistent with the needs of the hydraulic analysis.
#'
#' @return matrix, damages for each rooms and externals based on water level
#'
#' @examples
#'
#' model = adu_t
#'
#' # generate limnigraph
#' flood = generate_limnigraph(
#' 	 model = model,
#'   time = c(0, 300, 900),
#'   depth = cbind(facade = c(0, 3, 0)),
#'   exposition = list(
#'     facade = list(external = c("wall_A", "wall_B", "wall_C", "wall_D", "wall_E",
#' 		"wall_F", "wall_G", "wall_H")))
#' )
#' # calculate maximum flood water depth for event in "flood" in model adu_t
#' hydraulic = analyse_hydraulic(
#' 	model = model,
#' 	limnigraph = flood,
#' 	sim_id = "test",
#' 	stage = c("hydraulic")
#' )
#' # calculate damage for flood water depth calculated with hydraulic model
#' get_hydraulic_damage(
#'  h_max_by_room = hydraulic$hmax["z",], 
#'  damage = model[["dam_room_wall"]][c("absolute_room","absolute_external")], 
#'  exposition = flood[["exposition"]]
#' )
#'
#' # Test of some thresholds
#' hmax = c(room_1 = 0, room_1 = 0.005, room_1 = 0.2, room_1 = 0.12, room_1 = 10.0)
#' dam = list()
#' dam[["absolute_room"]][["room_1"]][["total"]] = matrix(
#'  c(0, 0, 0, 0, 1000, 2000, 0, 1500, 2500),
#'  ncol = 3, byrow = FALSE,
#'  dimnames = list(h = c(0, 10, 20), d = c(0, 12, 40))
#' )
#' get_hydraulic_damage(hmax, dam, NULL)
#'
#' @export

get_hydraulic_damage = function(
    h_max_by_room,
    damage, 
    exposition,
    threshold_height = 1,
    threshold_duration = 12
) {
    # extracting damage functions
    damage_room = damage[["absolute_room"]]
    damage_wall = damage[["absolute_external"]]

    # removing mean water height from calculation and convert to cm
    hmax = h_max_by_room[names(h_max_by_room) != "mean"] * 100

    damage_max = matrix(
        0, nrow = 1, ncol = length(hmax) + 1,
        dimnames = list("damage", c(names(hmax), "total"))
    )

    # checking for negative water level
    hmax = pmax(hmax, 0)

    # Ignore damage when water height is lower than or equal to threshold_height
    hmax[hmax <= threshold_height] = 0

    # Prepare h_sim and h_sim_max with one case (all cases should be the same)
    h_sim = as.integer(dimnames(damage[[1]][[1]][["total"]])[["h"]])
    h_sim_max = max(h_sim)
    hmax = pmin(hmax, h_sim_max)

    # selection of minimum available duration under threshold_duration
    duration = as.integer(dimnames(damage[[1]][[1]][["total"]])[["d"]])
    d = which(duration == min(duration[duration > 0], threshold_duration))
    if (length(d) == 0) stop(sprintf(
        "No duration %s available in damagings.", 
        threshold_duration
    ))

    for (i in seq_along(hmax)) {
        r = names(hmax)[i]
        # find wall damage for externals
        if (grepl("facade", r)) {
            for (e in exposition[[r]]) {
                for(w in e) {
                    damage_max["damage", i] = damage_max["damage", i] +
                        stats::approx(
                            h_sim,
                            damage_wall[[w]][["total"]][, d],
                            hmax[i]
                        )[["y"]]
                }
            }
        } else {
            # damage for rooms
            damage_max["damage", i] = stats::approx(
                h_sim,
                damage_room[[r]][["total"]][, d],
                hmax[i]
            )[["y"]]
        }
    }

    damage_max["damage", "total"] = sum(damage_max["damage", ])

    return(damage_max)
}
