#' Extract some values information of a model
#' 
#' @param model an object of class model
#' @param room a data.frame describing room of a model, set to
#'      `model[["data_table"]][["room"]]` if model is provided.
#' @param furniture a data.frame describing furniture of a model, set to
#'      `model[["data_table"]][["furniture"]]` if model is provided.
#' @param category a character vector describing category of a model, set to
#'      `model[["category"]]` if model is provided.
#' @param model_name a character scalar giving names of a model, set to
#'      `model[["name"]]` if model is provided.
#' @param verbose logical, default to getOption("floodam_building_verbose")
#' @return An list of value organized as follow
#' \describe{
#'   \item{surface}{a data.frame with 'internal' and 'external' surfaces.
#'          Surfaces are estimated at storey level, maximum is taken.}
#'   \item{monetary}{a data frame with the estimation of total monetary value
#'      replacement for elementary components classified  according to the 'category'
#'      of the model}
#' }
#' @examples
#' # Example 1
#' 
#' # analyzing model 'simple' of type 'simple' using the sequential steps *load* 
#' # (load model data) and *extract* (extract model data)
#' model = analyse_model(
#'      model = "adu_simple",
#' 		type = "adu",
#'      path = init_create_path(
#'          input = system.file("extdata", package = "floodam.building"),
#'          output = tempdir()
#'      ),
#'      stage = c("load", "extract")
#' )
#' extract_value(model)
#' 
#' # Example 2
#' 
#' # using already loaded library model 'dwelling' 
#' model = adu_t
#' 
#' # extracting model data
#' model = analyse_model(model = model, stage = "extract")
#' 
#' extract_value(model)
#' 
#' @export

extract_value = function(
        model,
		room = NULL,
		furniture = NULL,
		category = NULL,
		model_name = NULL,
		verbose = getOption("floodam_building_verbose")
	) 
{
    if (!missing(model)) {
        room = model[["data_table"]][["room"]]
        furniture = model[["data_table"]][["furniture"]]
        category = utils::head(model[["category"]], 1)
        model_name  = model[["name"]]
    }

	if (verbose) message(sprintf("Computing some values for '%s'...\n", model_name))

	# Computing some values at model level
    surface = data.frame(
        external = max(
			tapply(
				room[["surface"]][grep("external", room[["room"]])],
                room[["storey"]][grep("external", room[["room"]])],
				sum
			)
		),
        internal =  max(
			tapply(
				room[["surface"]][grep("room", room[["room"]])],
                room[["storey"]][grep("room", room[["room"]])],
				sum
			)
		)
    )
    surface[["ratio"]] = surface[["internal"]] / surface[["external"]]

	if (!is.null(furniture)) {
		monetary = stats::aggregate(
			furniture["value"],
			furniture[category],
			sum,
			na.rm = TRUE,
			drop = FALSE
		)
		monetary = as.data.frame(
			structure(
				as.list(monetary[["value"]]),
				names = as.character(monetary[[category]])
			)
		)
	} else {
		level = unique(floodam.building::COMPONENT_ELEMENTARY[[category]])
		monetary = as.data.frame(
			matrix(0, ncol = length(level), dimnames = list("value", level))
		)		
	}

	result = list(
		surface = surface,
		monetary = monetary
	)

	if (verbose) {
		message(sprintf("\t... Informations successfully extracted for '%s'\n", model_name))		
	}
	invisible(result)
}