#' Extract hydraulic input data from model
#' 
#' @param model an object of class model
#' @param initial_water_depth_interior numeric, initial water depth in each room in meters, default to
#' 		  getOption("floodam_building_hydraulic_initial_water_depth_interior")
#' @param discharge_coefficient numeric, discharge coefficient for each opening, default to
#' 		  getOption("floodam_building_hydraulic_discharge_coefficient")
#' @param clearance numeric, measurement of the opening gap clearance, default to
#' 		  getOption("floodam_building_hydraulic_close_opening_clearance")
#' @param verbose boolean, will floodam tells what it is doing, default to getOption("floodam_building_verbose")
#' @return component hydraulic of the model
#' 
#' @export

extract_hydraulic_input = function(
		model,
		initial_water_depth_interior = getOption("floodam_building_hydraulic_initial_water_depth_interior"),
		discharge_coefficient =	getOption("floodam_building_hydraulic_discharge_coefficient"),
	 	clearance = getOption("floodam_building_hydraulic_close_opening_clearance"),
		verbose = getOption("floodam_building_verbose")
	)
{	
	if (verbose) message(sprintf("Extracting input data for hydraulic model for '%s'...\n", model[["name"]]))	
	
	opening = levels(model[["data_table"]][["opening"]][["name"]])
	if (any(is.na(opening))) {
		if (verbose) message("\t... Openings not named. Cannot determine hydraulic model inputs.\n\n")
        return(NULL)
	}
	
	model[["hydraulic"]] = list()
	# determining exchanges for openings (open & close)
	exchange_opening = lapply(
		opening, 
		find_exchange, 
		x = model[["data_table"]][["opening"]]
	)
	exchange_opening_open = Reduce(rbind, exchange_opening)
	exchange_opening_open[["discharge_coeff"]] = discharge_coefficient
	exchange_opening_open[["clearance_under"]] = clearance["under"]
	exchange_opening_open[["clearance_width"]] = clearance["side"]

	exchange_opening_close = exchange_opening_open
	exchange_opening_close[["height"]] = exchange_opening_close[["clearance_under"]]

	exchange_opening_combine = exchange_opening_open
	exchange_opening_combine[
		grepl("^door|^window", exchange_opening_combine[["exchange"]]), 
		"height"
	] = exchange_opening_combine[
			grepl("^door|^window", exchange_opening_combine[["exchange"]]), 
			"clearance_under"
		]
	
	# storing exchanges
	model[["hydraulic"]][["exchanges_open"]] = exchange_opening_open
	model[["hydraulic"]][["exchanges_close"]] = exchange_opening_close
	model[["hydraulic"]][["exchanges_combine"]] = exchange_opening_combine

	# building table of surfaces and heights of rooms
	model[["hydraulic"]][["rooms"]] = model[["data_table"]][["room"]][
    	model[["data_table"]][["room"]][["room"]]!="external", 
    	c("room", "surface", "H_abs")
    ]
	model[["hydraulic"]][["rooms"]][["initial_depth"]] = initial_water_depth_interior

	# converting heights and widths to meters. Necessary for hydraulic model 
	if (verbose) message(sprintf("\t... converting hydraulic input data in '%s' to meters\n", model[["name"]]))
	
	selection_exchange = c("width", "H_abs", "height")
	selection_room = c("H_abs")

	model[["hydraulic"]] = sapply(
		names(model[["hydraulic"]]),
		function(x) {
			temp = model[["hydraulic"]][[x]]
			if(x == "rooms") {
				temp[ , selection_room] = temp[ , selection_room] / 100
			} else {
				temp[ , selection_exchange] = temp[ , selection_exchange] / 100
			}
			return(temp)
		},
		simplify = FALSE
	)

	if (verbose) message(sprintf("\t... hydraulic input data in '%s' succesfully converted to meters\n", model[["name"]]))

	if (verbose) message(sprintf("\t... hydraulic input data successfully extracted for '%s'\n", model[["name"]]))

	invisible(model[["hydraulic"]])
}

find_exchange = function(x, opening) {
    x = x[x[["name"]] %in% opening, ]
    exchange = paste0(x[["room"]], collapse = "|")
    if ("external" %in% x[["room"]]) {
        temp = x[["room"]][x[["room"]] != "external"]
        exchange = sprintf("%s|%s", unique(x[["name"]]), temp)
    }
    selection = unique(x[c("width", "H_abs", "height", "x", "y")])
    result = data.frame(exchange, selection)
    return(result)
}

#' Extract hydraulic input data from model
#' 
#' @param model an object of class model
#' @param exposition list, contains list of `facade` exposed to the flood event.
#' 	Each `facade` is a list of named vectors of the walls exposed in the same
#' 	way. Vectors are named after the `external` they belong to in the model.
#' @param initial_water_depth_interior numeric, initial water depth in each room in meters, default to
#' 		  getOption("floodam_building_hydraulic_initial_water_depth_interior")
#' @param discharge_coefficient numeric, discharge coefficient for each opening, default to
#' 		  getOption("floodam_building_hydraulic_discharge_coefficient")
#' @param clearance numeric, measurement of the opening gap clearance, default to
#' 		  getOption("floodam_building_hydraulic_close_opening_clearance")
#' @param permeability, character, optional; if water goes through the `side` of
#' 		the opening or `under` or `both`. Default to `under`, can be `side` or
#' 		`both`
#' @param verbose boolean, will floodam tells what it is doing, default to getOption("floodam_building_verbose")
#' @return component hydraulic of the model
#' 
#' @details
#' 
#' This function creates data.frames with data on the openings for the hydraulic module. It creates one for each scenario `open` `close` `combine`
#' 
#' @export

prepare_hydraulic_input = function(
		model,
		exposition,
		initial_water_depth_interior = getOption("floodam_building_hydraulic_initial_water_depth_interior"),
		discharge_coefficient =	getOption("floodam_building_hydraulic_discharge_coefficient"),
	 	clearance = getOption("floodam_building_hydraulic_close_opening_clearance"),
		permeability = c("under", "side", "both"),
		verbose = getOption("floodam_building_verbose")
	)
{
	# get opening frames and characteristics from the model
	var_sel = c("width", "H_abs", "height")
	opening = model[["data_table"]][["opening"]][c("name", "room", var_sel)]
	opening[var_sel] = opening[var_sel] / 100

	# create a data.frame for the hydraulic module
	result_frame = do.call(
		rbind,
		tapply(
			opening[["room"]],
			opening[["name"]],
			function(x){stats::setNames(sort(as.character(x)), c("up", "down"))}
		)
	)
	result_frame = cbind(opening = rownames(result_frame), result_frame)

	# new expositions are openings on the matching walls
	exposition_op = list()
	for (f in names(exposition)) {
		exposition_op[[f]] = c()
		for(e in names(exposition[[f]])) {
			exposition_op[[f]] = c(exposition_op[[f]], 
				as.character(model[["data_table"]][["opening"]][["name"]][
				model[["data_table"]][["opening"]][["id_wall"]] %in%
				exposition[[f]][[e]]]))
		}
	}

	# set connections for rooms
	if (!missing(exposition_op)){
		for (f in names(exposition_op)) {
			sel = grep("^external", result_frame[exposition_op[[f]], "up"])
			result_frame[exposition_op[[f]], "up"][sel] = f
			sel = grep("^external", result_frame[exposition_op[[f]], "down"])
			result_frame[exposition_op[[f]], "down"][sel] = f
		}
	}

	# initialize df for all the scenarios with the open frames
	op_open = op_close = op_combine = cbind(
		merge(
			result_frame,
			unique(opening[c("name", var_sel)]),
			by.x = "opening",
			by.y = "name"
		),
		discharge_coeff = discharge_coefficient
	)

	# create new empty openings for the open case if permeability is `both`
	if (permeability == "both") {
		op_open_width = op_open
		op_open_width[["opening"]] = paste0(op_open_width[["opening"]], "_v")
		op_open_width[["height"]] = 0.0
		op_open = rbind(op_open_width, op_open)
	}

	# create new openings to combine vertical and horizontal permeability
	if (permeability == "both")
	{
		op_close_width = op_close_height = op_close
		op_close_width[["opening"]] = paste0(op_close_width[["opening"]], "_v")
		op_close_width[["width"]] = clearance["under"] / 100
		op_close_height[["height"]] = clearance["side"] / 100
		op_close = rbind(op_close_width, op_close_height)
	} else {
		side = if (permeability == "side") "width" else "height"
		op_close[[side]] = clearance / 100
	}

	# find openings connected to the exterior
	op_ext = grepl("^facade", op_combine[["up"]]) | grepl("^facade",
		op_combine[["down"]])

	# create new openings to combine vertical and horizontal permeability
	if (permeability == "both") {
		op_combine_width = op_combine[op_ext, ]
		op_combine_width[["opening"]] = paste0(op_combine_width[["opening"]],
			"_v")
		op_combine[["height"]][op_ext] = clearance["under"] / 100
		op_combine_width[["width"]] = clearance["side"] / 100
		op_combine = rbind(op_combine_width, op_combine)

		name = op_combine[["opening"]]
		op_open = op_open[op_open[["opening"]] %in% name, ]
	} else {
		op_combine[[side]][op_ext] = clearance / 100
	}

	room = model[["data_table"]][["room"]][c("room", "surface", "H_abs", "ceiling_H")]
	room[["H_abs"]] = room[["H_abs"]] / 100
	room = room[grepl("^room", room[["room"]]), ]
	room[["initial_depth"]] = initial_water_depth_interior / 100

	list(
		open = op_open,
		close = op_close,
		combine = op_combine,
		room = room
	)
}