#' @title Extract elementary damages by room
#' 
#' @param x the list of elementary damaging in a model, as given in 
#' the slot "detail" of the slot "damaging" of a model
#' @param room character, names of the rooms in the model
#' 
#' @details This function is meant to be called during the calculation of damage
#'  functions
#' 
#' @return a list 
#' 
#' @examples
#' model = analyse_model(
#'  model = adu_t_basement,
#'  stage = c("extract", "damaging")
#' )
#' 
#' test = extract_damage_detail_by_room(
#'  x = model[["damaging"]][["detail"]], 
#'  room = levels(model[["data_table"]][["room"]][["room"]])
#' )
#' 
#' @seealso [extract_elementary_component_from_detail()] and 
#' [extract_damage_detail_external()], to handle the calculation of 
#' damage functions by rooms and walls
#' 
#' @export 
#' 

extract_damage_detail_by_room = function(x, room){
  
  # filtering potential null slots in x
  selection = !unlist(lapply(x, is.null))
  selection = names(x[selection])
  
  # initializing liqt
  result = list()

  # extracting elementary components from detail by room
  for (r in room){
    result[[r]] = sapply(
      selection,
      FUN = extract_elementary_component_from_detail, 
      x = x, 
      room = r, 
      simplify = FALSE
    )
  }

  if (isFALSE(any(grepl("external", names(result))))) {
    warning("The 'external*' slot has not been found. The damage by external wall is not calculated")
    return(result)
  }

  #  returning result
  return(result)
}

#' @title Extract elementary damages by categories of elementary components
#' 
#' @description 
#' Extract elementary damages by categories of elementary components (floor,
#' ceiling, wall, opening, coating and furniture).
#' 
#' @param x list of `elementary_damaging` objects
#' @param what character, the name of the category to be extracted
#' @param room character, name of the room for which `elementary_damaging` will
#'  be extracted.
#' 
#' @details This function is meant to be called during the calculation of damage
#'  functions
#' 
#' @return a list of `elementary_damaging` objects
#' 
#' @examples
#' # example for the category "wall" in "room_1"
#' test = extract_elementary_component_from_detail(
#'  x = adu_t[["damaging"]][["detail"]], 
#'  what = "wall",
#'  room = "room_1"
#' )
#' 
#' @seealso [extract_damage_detail_by_room()] and 
#' [extract_damage_detail_external()], to handle the calculation of 
#' damage functions by rooms
#' 
#' @export

extract_elementary_component_from_detail = function(x, what, room){
  
  # Since the "room label" is different for "furniture" we set up a variable 
  # depending on the value of the parameter 'what' 
  # if (what == "furniture"){
  #   label = "room_name"
  # } else {
  #   label = "room"
  # }
  label = "room"
  
  # creating sequence with all the elements in the slot 'what' of 'x'
  id = seq_along(names(x[[what]]))
  
  # scrolling the list and retaining elements in the room indicated in 'room'
  selection = unlist(
    lapply(
      id, 
      function(id){
        attr(x[[what]][[id]], "detail")[[label]] == room
      }
    )
  )

  result = x[[what]][selection]

  #  returning result
  if (!length(result)) {
    return(NULL)
  } else {
    return(result)
  }
}

#' @title Extract elementary components in external by wall
#' 
#' @description 
#' `extract_damage_detail_external()` is an **internal** function that
#' extracts elementary components in external by wall.
#' 
#' @param x list of elementary damaging corresponding to the external wall of 
#'  a given storey organized and named as given in the slot "detail" of the slot
#' "damaging" of a model
#'  
#' @param wall character, the names of the external walls in the storey
#' 
#' @details This function is meant to be called during the calculation of damage
#'  functions
#' 
#' @return a list of elementary components 
#' 
#' @examples 
#'  # example 1
#'  extract_damage_detail_external(
#'    x = adu_t[["damaging"]][["detail"]], 
#'    wall = "wall_A"
#'  )
#' 
#'  #example 2
#'  test = analyse_model(adu_t_basement, stage = c("extract", "damaging"))
#'  data_table = test$data_table
#'  result = test$damaging
#' 	selection = grepl(
#'    "external",
#'    levels(data_table[["wall"]][["room"]])
#'  )
#'
#'  external = split(
#'    as.character(data_table[["wall"]][["wall"]]), 
#'    data_table[["wall"]][["room"]]
#'  )[selection]
#'
#'  detail_by_room = extract_damage_detail_by_room(
#'    x = result[["detail"]], 
#'    room = levels(data_table[["room"]][["room"]])
#'  )
#'
#'  detail_by_external = sapply(
#'      names(external),
#'    function(ew) {
#'      extract_damage_detail_external(
#'        detail_by_room[[ew]], 
#'        wall = external[[ew]]
#'      )
#'    },
#'    simplify = FALSE
#'  )
#' 
#' @seealso [extract_damage_detail_by_room()] and 
#' [extract_elementary_component_from_detail()], to handle the calculation of 
#' damage functions by rooms and walls
#' 
#' @export

extract_damage_detail_external = function(x, wall) {

  result = sapply(
    wall,
    function(segment, x) {
        result = list()
        for (what in names(x)){
          # choose column name depending on name slot
          # if (what == "furniture"){
          #     label = "wall_name"
          # } else {
          #     label = "wall"
          # }
          label = "wall"
          
          # count slots and choose whether to end the function directly or 
          # proceed with extraction
          id = seq_along(names(x[[what]]))
          if (is.null(id)) {
              result[[what]]= NULL
          } else {
            selection = c()
            for (i in id) {
                presence = attr(x[[what]][[i]], "detail")[[label]] == segment
                selection = c(selection, presence)
            }
            selection[is.na(selection)] = FALSE
            if (any(selection)) {
              result[[what]] = x[[what]][selection]
            } else {
              result[[what]] = NULL
            }
          }
        }

        # check result's length and return NULL if empty
        if (length(result) != 0) {
          return(result)
        } else {
          return(NULL)
        }
    },
    x = x,
    simplify = FALSE
  )

  return(result)
}
