#' Creates declination of model according to the declination file
#
#' @param model the selected model to decline
#' @param combination table with the characteristics of each version to create
#' @param meas boolean saying if adaptation measure should be applied or not
#' @param selection the rooms where material measures will be applied
#' @param core number of cores allocated to the creation of the versions, defaults to one
#' @return a list with the new xmls and damaging
#' 
#' @keywords internal

create_new_building = function(
    model,
    meas = FALSE,
    selection = NULL,
    combination,
    core = 1
  )
{

  core = max(core, parallel::detectCores())
  
  #Functions changing elements
  change_wall = function(element, material){
	if (!is.null(material[ , paste("wall", gsub("_wall", "", element[["wall_type"]]), "material", sep = "_")])){
	element[["material"]] = as.character(material[ , paste("wall", gsub("_wall", "", element[["wall_type"]]), "material", sep = "_")])
	}
	if (!is.null(element[["render"]]) & !is.na(material[["wall_load_bearing_render"]])){
	element[["render"]] = as.character(material[["wall_load_bearing_render"]])
	} else {element[["render"]] = NULL}
	
	if (!is.null(element[["insulating"]]) & !is.na(material[["wall_lining_insulating"]])){
	element[["insulating"]] = as.character(material[["wall_lining_insulating"]])
	} else{element[["insulating"]] = NULL}
	return(element)
}

  change_coating = function(element, material){
    if (!is.null(material[["coating_material"]])){
      element[["material"]] = as.character(material[["coating_material"]])
    }
    return(element)
  }
  
  change_opening = function(element, material){
    if (element[["opening_type"]] != "door" & !is.null(material[["opening_window_material"]])){
      element[["material"]] = as.character(material[["opening_window_material"]])
      element[["shutter"]] = as.character(material[["opening_window_shutter"]])
    } else if (length(grep("garage", element[["material"]])) == 1 & !is.null(material[["opening_door_garage_material"]])){
      element[["material"]] = as.character(material[["opening_door_garage_material"]])
    } else if (length(grep("exterior", element[["material"]])) == 1 & !is.null(material[["opening_door_ext_material"]])){
      element[["material"]] = as.character(material[["opening_door_ext_material"]])
    } else if (length(grep("entrance", element[["material"]])) == 1 & !is.null(material[["opening_door_ext_material"]])){
      element[["material"]] = as.character(material[["opening_door_ext_material"]])
    } else if (element[["opening_type"]] == "door" & length(grep("garage", element[["material"]])) != 1
               & length(grep("exterior", element[["material"]])) != 1 & length(grep("entrance", element[["material"]])) != 1
               & !is.null(material[["opening_door_int_material"]])){
      element[["material"]] = as.character(material[["opening_door_int_material"]])
    }
    return(element)
  }

  change_floor = function(element, material){
    if (!is.null(material[["floor_material"]])){
      element[["material"]] = as.character(material[["floor_material"]])
      element[["coating"]] = as.character(material[["floor_coating"]])
    }
    return(element)
  }
  
  change_ceiling = function(element, material){
    if (!is.null(material[["ceiling_material"]])) {
      element[["material"]] = as.character(material[["ceiling_material"]])
      element[["coating"]] = as.character(material[["ceiling_coating"]])
      if (!(material[["ceiling_material"]] %in% c("concrete", "glued_mineral", "glued_polystyrene")) & !is.na(material[["ceiling_insulating"]])){
        element[["insulating"]] = as.character(material[["ceiling_insulating"]])
      } else element[["insulating"]] = NULL
    }
    return(element)
  }
  
  create_version = function(i){ 
    #Change material
    material = combination[i,]
    model_new = model
    
    model_new[["building"]][["element"]][["wall"]] = lapply(model_new[["building"]][["element"]][["wall"]], change_wall, material)
    model_new[["building"]][["element"]][["coating"]] = lapply(model_new[["building"]][["element"]][["coating"]], change_coating, material)
    model_new[["building"]][["element"]][["opening"]] = lapply(model_new[["building"]][["element"]][["opening"]], change_opening, material)
    model_new[["building"]][["element"]][["floor"]] = lapply(model_new[["building"]][["element"]][["floor"]], change_floor, material)
    model_new[["building"]][["element"]][["ceiling"]] = lapply(model_new[["building"]][["element"]][["ceiling"]], change_ceiling, material)
    if ("dilapidation" %in% names(combination)){
      model_new[["building"]][["general"]][["dilapidation"]] = as.numeric(as.character(material[["dilapidation_value"]]))
    }
    
    #Update the tables with the substitute elements
    model_new[["damaging"]] = compute_damage(model_new)
    model_new[["furniture"]] = model[["furniture"]]
    model_new[["surface"]] = sum(model[["room"]][["surface"]][!(model[["room"]][["room"]] %in% grep("external", model[["room"]][["room"]], value = TRUE)) & model[["room"]][["H_abs"]] == 0])
    model_new[["version"]] = paste("V", i, sep = "_")

    #Apply vulnerability reduction measures
    if (meas == TRUE){
      adaptation = apply_measure(model_new, "material", selection, graph = FALSE)
      adaptation[["surface"]] = model_new[["surface"]]
      return(adaptation)
    }
    
    if (meas == FALSE){
    return(model_new)
    }
  }
  
  list_model = parallel::mclapply(1:nrow(combination), create_version, mc.cores = core)
  return(list_model)
}
