#' Analyse the dangerosity of the simulation
#'
#' @param result_h matrix containg water level in rooms associated with their 
#'      time
#' @param opening hydraulic part of the model, containing informations on the 
#' 		exchange
#' @param room hydraulic part of the model, containing informations on the 
#' 		rooms
#' @param bound numeric, matrix of floodwater depth heights by instant t
#' 		for each exterior opening exposed to flooding
#' @param dt_max numeric, optional; maxmimum timestep for simulation
#' @param t_max numeric, duration of the flood event in seconds,
#' 		optional; if not provided, calculated using limnigraph
#'
#' @return matrix of dangerosity for all openings
#'
#' @export

compute_danger = function(result_h, opening, room, bound, dt_max, t_max) {

    exterior_name = colnames(bound)[-1]
	room_name = room[["room"]] = as.character(room[["room"]])
    opening_name = opening[["opening"]]
	building_name = c(room_name, exterior_name)

	# init building and boundaries
	building = rbind(
		surf = c(room[["surface"]],	rep(NA_real_, ncol(bound) - 1)),
		z0 = c(room[["H_abs"]], numeric(ncol(bound) - 1)),
		z = c(room[["initial_depth"]], bound[1, -1])
	)
	colnames(building) = building_name

    # init exchange
    exchange = rbind(
        width = opening[["width"]],
        z_bottom = opening[["H_abs"]],
        z_top = opening[["H_top"]], 
        Cd = opening[["discharge_coeff"]],
        z_up = 0,
        z_down = 0,
        flow = 0,
        s = 0
    )
    colnames(exchange) = opening_name

    # setting up the relations between rooms and exchange
	match_opening = t(as.matrix(opening[c("up", "down")]))
	colnames(match_opening) = opening_name
	matching = matrix(
		0, 
		nrow = nrow(opening),
		ncol = ncol(building),
		dimnames = list(opening_name, building_name)
	)
	for (o in opening_name) matching[o, match_opening[, o]] = c(-1, 1)

    # preallocating the output matrix
    nrows = floor(t_max / dt_max)
    result_dg = matrix(
        NA_real_,
        nrow = nrows,
        ncol = nrow(opening) + 1,
        dimnames = list(c(), c("time", opening_name)))

    # computing dangerosity
    for(i in seq_along(result_h[, "time"])) {
        exchange[c("z_up", "z_down"), ] = result_h[i, match_opening]
        exchange[c("s", "flow"), ] = rcpp_compute_exchange(exchange)
        D = abs(exchange["flow", ] / exchange["width", ])
        result_dg[i, ] = c(result_h[i, "time"], D)
    }

    return(result_dg)
}
