#' @title Compute damage by room and segment of external wall in a given model
#' 
#' @description 
#' This function computes damage by room and external wall segment in a provided
#' model. It calculates damage by, first, grouping elementary components by room
#' and segment of external, and calculating the absolute damage for the given  
#' hazard range
#' 
#' @param model model
#' @param hazard_range list, default to
#' 	getOption("floodam_building_hazard_range")
#' @param dam list, default to floodam.building::DAM
#' @param h_abs character, default to "room". See details
#' @param verbose logical, default to getOption("floodam_building_verbose")
#' @return a list containing the computed absolute damage by room and external 
#'   wall segment.  The list has two elements:
#'   * `absolute_room`: A list where each element represents the damage 
#'     for a specific room.
#'   * `absolute_external`: A list where each element represents the damage 
#'     for a specific external wall segment.
#' 
#' @details 
#' The function has only been tested for version 02 of building input file.
#' 
#' The h_abs parameter is used to control whether calculations are performed
#' using the original level at which the room and external walls are located in
#' the building (h_abs = "model"), or an alternative level that disregards the
#' room's floor level in relation to street level and sets the room's floor to
#' zero (h_abs = "room").
#' 
#' @examples
#' 
#' model = analyse_model(adu_t_basement, stage = "extract")
#' model[["dam_room_wall"]] = compute_damage_by_room_wall(
#' 	model = model
#' )
#' 
#' model[["dam_room_wall"]] = compute_damage_by_room_wall(
#' 	model = model,
#'  h_abs = "model"
#' )
#' 
#' @export

compute_damage_by_room_wall = function(
	model = model,
	hazard_range = getOption("floodam_building_hazard_range"),
	dam = floodam.building::DAM,
	h_abs = "room",
	verbose = getOption("floodam_building_verbose")
)
{

	# extracting neccesary information from model
	model_name = model[["name"]]
	category = model[["category"]]
	version_building = model[["building"]][["general"]][["version"]]
	dilapidation = model[["building"]][["general"]][["dilapidation"]]
	file_log = file.path(model[["path"]][["model_log"]], model[["file_name"]][["log"]])
	data_table = model[["data_table"]]

	# checks and information to user
	if (isTRUE(verbose)) {
        message(sprintf("Computing damage by room and external wall segment for '%s'...", model_name))
    }

	if (!(version_building %in% sprintf("version_0%s",1:3))) {
		stop(sprintf("No compute_damage implementation available for %s of building", version_building))
	}
	if (version_building != "version_02") {
		warning("function untested for your version of building. Errors may appear")
	}

	# Detail damaging
	detail = compute_damage_detail(
		data_table = data_table,
		dam = dam,
		hazard_range = hazard_range,
		h_abs = h_abs,
		file_log = file_log,
		dilapidation = dilapidation,
		version_building = version_building,
		talkative = FALSE
	)

	# Absolute damaging by room and external
	selection = grepl(
		"external",
		levels(data_table[["wall"]][["room"]])
	)

	external = split(
		as.character(data_table[["wall"]][["wall"]]), 
		data_table[["wall"]][["room"]]
	)[selection]

	external_id = split(
		rownames(data_table[["wall"]]), 
		data_table[["wall"]][["room"]]
	)[selection]

	detail_by_room = extract_damage_detail_by_room(
		x = detail, 
		room = levels(data_table[["room"]][["room"]])
	)

	detail_by_external = lapply(
		names(external),
		function(ew, external_id) {
			temp = extract_damage_detail_external(
				detail_by_room[[ew]], 
				wall = external[[ew]]
			)
			names(temp) = external_id[[ew]]
			return(temp)
		},
		external_id = external_id
	)

	detail_by_external = unlist(detail_by_external, recursive = FALSE)

	# calculating damage
	result = list()
	for (room in names(detail_by_room)){
		result[["absolute_room"]][[room]] = compute_damage_from_detail(
			x = detail_by_room[[room]], 
			category = category,
			hazard_range = hazard_range
		)
	}

	for (wall in names(detail_by_external)) {
		result[["absolute_external"]][[wall]] = compute_damage_from_detail(
			x = detail_by_external[[wall]], 
			category = category,
			hazard_range = hazard_range
		)               

	}		

	if (isTRUE(verbose)) {
        message(
            sprintf(
                "\t... Damaging by room and external wall successfully computed for '%s'",
                model_name
            )
        )
    }
 
	invisible(result)
}