#' Compute damage for a given model
#' 
#' @param model model
#' @param hazard_range list, default to
#' 	getOption("floodam_building_hazard_range")
#' @param verbose logical, default to getOption("floodam_building_verbose")
#' @param dam list, default to floodam.building::DAM
#' @return component damaging of the model
#' 
#' @export

compute_damage = function(
	model,
	hazard_range = getOption("floodam_building_hazard_range"),
	dam = floodam.building::DAM,
	verbose = getOption("floodam_building_verbose")
)
{
	version_building = model[["building"]][["general"]][["version"]]
	compute = switch(
		EXPR = version_building,
		"version_03" = compute_damage_03,
		"version_02" = compute_damage_02,
		"version_01" = compute_damage_01,
		stop(sprintf("No compute_damage implementation available for version %s of building", version_building))
	)

	return(
		compute(
			model[["data_table"]],
			hazard_range = hazard_range,
			dam = dam,
			dilapidation = model[["building"]][["general"]][["dilapidation"]],
			category = model[["category"]],
			value = model[["value"]],
			model_name = model[["name"]],
			file_log = file.path(
                model[["path"]][["model_log"]],
                model[["file_name"]][["log"]]
            ),
			verbose = verbose
		)
	)
}

compute_damage_03 = function(
	data_table,
	hazard_range = getOption("floodam_building_hazard_range"),
	dam = floodam.building::DAM,
	dilapidation = getOption("floodam_building_dilapidation"),
	category = NULL,
	value = NULL,
	model_name = NULL,
	file_log = NULL,
	verbose = getOption("floodam_building_verbose"),
	talkative = getOption("floodam_building_talkative")
)
{
	version_building = "version_03"

	if (isTRUE(verbose)) {
        message(sprintf("Computing damage for '%s'...\n", model_name))
    }

  	# Check integrity between elementary components and damaging
	missing_component = check_damaging(data_table, file_log, names(dam))
 	if (isTRUE(verbose) && length(missing_component > 0)) {
		message(
            "\t... ",
            sprintf(
                "Damaging computed ignoring %s missing elementary components",
                length(missing_component)
            )
        )
	}

	detail = list()
	
	### Room
	# Preparing ceiling
	ceiling = data_table[["room"]][c(
		"ceiling_elementary_component", 
		"ceiling_insulating_elementary_component",
		"ceiling_coating_elementary_component",
		"H_abs", "surface",	"ceiling_H")]
	ceiling[["H_abs_room"]] = ceiling[["H_abs"]]
	ceiling[["H_abs"]] = ceiling[["H_abs"]] + ceiling[["ceiling_H"]]	
	detail[["room"]] = c(
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["room"]][["floor_elementary_component"]])],
			component = split(data_table[["room"]], seq(nrow(data_table[["room"]]))),
			MoreArgs = list(
				what = "floor_elementary_component",
				dilapidation = 0,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["room"]][["floor_coating_elementary_component"]])],
			component = split(data_table[["room"]], seq(nrow(data_table[["room"]]))),
			MoreArgs = list(
				what = "floor_coating_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["room"]][["floor_insulating_elementary_component"]])],
			component = split(data_table[["room"]], seq(nrow(data_table[["room"]]))),
			MoreArgs = list(
				what = "floor_insulating_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_elementary_component",
				dilapidation = 0,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_insulating_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_insulating_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_coating_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_coating_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE)
		)
	detail[["room"]] = detail[["room"]][!sapply(detail[["room"]], is.null)]
	
	# Furniture
	if (!is.null(data_table[["furniture"]])) {
		detail[["furniture"]] = c(
			mapply(adjust_elementary_damaging, 
				damaging = dam[as.character(data_table[["furniture"]][["component_elementary"]])],
				component = split(data_table[["furniture"]], seq(nrow(data_table[["furniture"]]))),
				MoreArgs = list(
					what = "component_elementary",
					dilapidation = 0, # Already included in furniture.csv !
					hazard_range = hazard_range,
					file_log = file_log),
				SIMPLIFY = FALSE)
			)
		detail[["furniture"]] = detail[["furniture"]][!sapply(detail[["furniture"]], is.null)]
	}

	# Openings
	if (!is.null(data_table[["opening"]])) {
		detail[["opening"]] = c(
			mapply(adjust_elementary_damaging, 
				damaging = dam[as.character(data_table[["opening"]][["opening_elementary_component"]])],
				component = split(data_table[["opening"]], seq(nrow(data_table[["opening"]]))),
				MoreArgs = list(
					what = "opening_elementary_component",
					dilapidation = dilapidation,
					hazard_range = hazard_range,
					file_log = file_log),
				SIMPLIFY = FALSE),
			mapply(adjust_elementary_damaging, 
				damaging = dam[as.character(data_table[["opening"]][["shutter_elementary_component"]])],
				component = split(data_table[["opening"]], seq(nrow(data_table[["opening"]]))),
				MoreArgs = list(
					what = "shutter_elementary_component",
					dilapidation = dilapidation,
					hazard_range = hazard_range,
					file_log = file_log),
				SIMPLIFY = FALSE)
			)
		detail[["opening"]] = detail[["opening"]][!sapply(detail[["opening"]], is.null)]
		detail[["opening"]] = lapply(detail[["opening"]], function(x){x/2}) 
	}
		
	# Walls
	surface_height = mapply(calculate_surface_height,
		wall = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
		MoreArgs = list(
            opening = data_table[["opening"]],
            version_building = version_building
        )
    )
	detail[["wall"]] = c(
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["wall_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "wall_elementary_component",
				dilapidation = 0, # No distinction between partition and load_bearing_wall (as done by CEPRI 2013)
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["insulating_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "insulating_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["render_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "render_elementary_component",
				dilapidation = dilapidation,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE)
		)
	detail[["wall"]] = detail[["wall"]][!sapply(detail[["wall"]], is.null)]
	selection = grep("partition", names(detail[["wall"]]))
	selection_names = grep("partition", names(detail[["wall"]]), value = TRUE)
	detail[["wall"]][selection] = lapply(detail[["wall"]][selection], function(x){x / 2}) 
	names(detail[["wall"]][selection]) = selection_names
	
	# Coating
	if (!is.null(data_table[["coating"]])) {
		surface_height = mapply(calculate_surface_height,
			wall = split(data_table[["coating"]], seq(nrow(data_table[["coating"]]))),
			MoreArgs = list(
                opening = data_table[["opening"]],
                version_building = version_building
            )
        )
		detail[["coating"]] = c(
			mapply(adjust_elementary_damaging, 
				damaging = dam[as.character(data_table[["coating"]][["coating_elementary_component"]])],
				component = split(data_table[["coating"]], seq(nrow(data_table[["coating"]]))),
				surface_height = surface_height,
				MoreArgs = list(
					what = "coating_elementary_component",
					dilapidation = dilapidation,
					hazard_range = hazard_range,
					file_log = file_log),
				SIMPLIFY = FALSE)
			)
		detail[["coating"]] = detail[["coating"]][!sapply(detail[["coating"]], is.null)]
	}

	result = list(detail = detail)

	# Absolute damaging
	component_category = levels(floodam.building::COMPONENT_ELEMENTARY[[category]])
	component_category = stats::setNames(
		component_category[component_category != "building"],
		component_category[component_category != "building"]
	)
	selection = list()
	for (what in component_category) {
		selection[[what]] = sapply(
            detail[["furniture"]],
            function(x){
                floodam.building::COMPONENT_ELEMENTARY[attr(x, "component_elementary"), category] == what
            }
        )
	}
	selection[["building"]] = !apply(as.data.frame(selection), 1, any)

	building  = detail[["furniture"]][selection[["building"]]]
	building = if (length(building) == 0) {
        detail[c("room", "opening", "wall", "coating")]
    } else {
        c(detail[c("room", "opening", "wall", "coating")], list(furniture = building))
    }

	result[["absolute"]] = list(total = sumup_damaging(detail), building = sumup_damaging(building))
	for (what in component_category) {
		result[["absolute"]][[what]] = 
            Reduce(
                "+" ,
                lapply(
                    detail[["furniture"]][selection[[what]]],
                    function(x){apply(x, 1:2, sum, na.rm = TRUE)}
                )
            )
		if (is.null(result[["absolute"]][[what]])) {
            result[["absolute"]][[what]] = 
                array(
                    0,
                    dim = dim(result[["absolute"]][["total"]]),
                    dimnames = dimnames(result[["absolute"]][["total"]])
                )
        }
	}
	
	# Relative damaging
	result[["relative"]] =
        list(
            building = result[["absolute"]][["building"]] / sum(unlist(value[["monetary"]][component_category]))
        )
	for (what in component_category) {
		value = if (is.null(value[[what]])) 0 else value[[what]]
		result[["relative"]][[what]] = result[["absolute"]][[what]] / value
	}
	
	# Check possible problem with relative damagement
	for (what in component_category) {
		if (any(selection[[what]])) {
			value_category = 
                sapply(
                    result[["detail"]][["furniture"]][selection[[what]]],
                    function(x){attr(x, "detail")[["value"]]}
                )
			warning_what =
                sapply(
                    result[["detail"]][["furniture"]][selection[[what]]],
                    max,
                    na.rm = TRUE
                ) / value_category > 1
			warning_what[is.na(warning_what)] = TRUE
			if (any(warning_what)) {
				msg = c(
                    sprintf(
                        "Possible problem with those %s components (relative damaging > 1 or value = 0):",
                        what
                    ),
					paste(
                        "\t-",
                        sapply(
                            result[["detail"]][["furniture"]][selection[[what]]][warning_what],
                            function(x){
                                paste(
                                    attr(x, "detail")[c("component_elementary", "room_name", "H_abs")],
                                    collapse = " / "
                                )
                            }
                        )
                    )
                )
				write(msg, file_log, append = TRUE)
				warning(
                    sprintf(
                        "Possible problem with some %s components (relative damaging > 1 or value = 0), see %s for details",
                        what,
                        file_log
                    )
                )
			}
		}
	}
	
	# Surface damaging
	result[["surface"]] = list(
		building = result[["absolute"]][["building"]] / (value[["surface"]][["internal"]])
	)
	
	if (isTRUE(verbose)) {
        message(
            sprintf(
                "\t... Damaging successfully computed for '%s'",
                model_name
            )
        )
    }
 
	invisible(result)
}

compute_damage_02 = function(
	data_table,
	hazard_range = getOption("floodam_building_hazard_range"),
	dam = floodam.building::DAM,
	dilapidation = getOption("floodam_building_dilapidation"),
	category = NULL,
	value = NULL,
	model_name = NULL,
	file_log = NULL,
	verbose = getOption("floodam_building_verbose"),
	talkative = getOption("floodam_building_talkative")
)
{
    version_building = "version_02"

	if (isTRUE(verbose)) {
        message(sprintf("Computing damage for '%s'...", model_name))
    }

  	# Check integrity between elementary components and damaging
	missing_component = check_damaging(data_table, file_log, names(dam))
 	if (isTRUE(verbose) && length(missing_component > 0)) {
		message(
            "\t... ",
            sprintf(
                "Damaging computed ignoring %s missing elementary components",
                length(missing_component)
            )
        )
	}

	result = list()
	
	# Detail damaging
	result[["detail"]] = compute_damage_detail(
		data_table = data_table,
		dam = dam,
		hazard_range = hazard_range,
		file_log = file_log,
		dilapidation = dilapidation,
		version_building = "version_02",
		talkative = talkative
	)

	# Absolute damaging
	if (isTRUE(talkative)) {
        message("\t6. Compute absolute damaging")
    }
	component_category = levels(
		as.factor(floodam.building::COMPONENT_ELEMENTARY[[category]])
	)
	component_category = stats::setNames(
		component_category[component_category != "building"],
		component_category[component_category != "building"]
	)
	selection = list()
	for (what in component_category) {
		selection[[what]] = sapply(
			result[["detail"]][["furniture"]],
			function(x){
                floodam.building::COMPONENT_ELEMENTARY[attr(x, "component_elementary"), category] == what
            }
		)
	}
	selection[["building"]] = !apply(as.data.frame(selection), 1, any)

	building  = result[["detail"]][["furniture"]][selection[["building"]]]
	building = if (length(building) == 0) {
		result[["detail"]][c("room", "opening", "wall", "coating")]
	} else {
		 c(result[["detail"]][c("room", "opening", "wall", "coating")], list(furniture = building))
	}

	result[["absolute"]] = list(
		total = sumup_damaging(result[["detail"]]),
		building = sumup_damaging(building)
	)
	for (what in component_category) {
		result[["absolute"]][[what]] = Reduce(
			"+" ,
			lapply(
				result[["detail"]][["furniture"]][selection[[what]]],
				#result[["detail"]][[what]],
				function(x){apply(x, 1:2, sum, na.rm = TRUE)}
			)
		)
		if (is.null(result[["absolute"]][[what]])) {
			result[["absolute"]][[what]] = array(
				0,
				dim = dim(result[["absolute"]][["total"]]),
				dimnames = dimnames(result[["absolute"]][["total"]])
			)
		}
	}

	# Relative damaging
	if (isTRUE(talkative)) {
        message("\t7. Compute relative damaging")
    }
	result[["relative"]] = list(
		building = result[["absolute"]][["building"]] / sum(unlist(value[["monetary"]][component_category]))
	)
	for (what in component_category) {
		result[["relative"]][[what]] = result[["absolute"]][[what]] / value[[what]]
	}
	
	# Check possible problem with relative damagement
	if (isTRUE(talkative)) {
        message("\t8. Check possible problem with relative damagement")
    }
	for (what in component_category) {
		if (any(selection[[what]])) {
			value_category = sapply(
				result[["detail"]][["furniture"]][selection[[what]]],
				function(x){attr(x, "detail")[["value"]]}
			)
			warning_what = 
				sapply(
					result[["detail"]][["furniture"]][selection[[what]]],
					max,
					na.rm = TRUE
				) / value_category > 1
			warning_what[is.na(warning_what)] = TRUE
			if (any(warning_what)) {
				msg = c(
					sprintf(
						"Possible problem with those %s components (relative damaging > 1 or value = 0):",
						what
					),
					paste(
						"\t-",
						sapply(
							result[["detail"]][["furniture"]][selection[[what]]][warning_what],
							function(x){
								paste(
									attr(x, "detail")[c("component_elementary", "room_name", "H_abs")],
									collapse = " / "
								)
							}
						)
					)
				)
				write(msg, file_log, append = TRUE)
				warning(
                    sprintf(
                        "Possible problem with some %s components (relative damaging > 1 or value = 0), see %s for details",
                        what,
                        file_log
                    )
                )
			}
		}
	}

	# Surface damaging
	result[["surface"]] = list(
		building = result[["absolute"]][["building"]] / (value[["surface"]][["internal"]])
	)

	if (isTRUE(verbose)) {
        message(
            sprintf(
                "\t... Damaging successfully computed for '%s'",
                model_name
            )
        )
    }
 
	invisible(result)
}

compute_damage_01 = function(
		data_table,
		hazard_range = getOption("floodam_building_hazard_range"),
		dam = floodam.building::DAM,
		dilapidation = getOption("floodam_building_dilapidation"),
		category = NULL,
		value = NULL,
		model_name = NULL,
		file_log = NULL,
		verbose = getOption("floodam_building_verbose"),
		talkative = getOption("floodam_building_talkative")
	)
{
	version_building = "version_01"
	
	sumup = function(x) {
		result = lapply(x, function(x){lapply(x, function(x){apply(x, 1:2, sum, na.rm = TRUE)})})
		Reduce("+", lapply(result, Reduce, f = "+"))
	}
	
	if (isTRUE(verbose)) {
        message(sprintf("Computing damage for '%s'...\n", model_name))
    }

  	# Check integrity between elementary components and damaging
	missing_component = check_damaging(data_table, file_log, names(dam))
 	if (isTRUE(verbose) && length(missing_component > 0)) {
		message(
            "\t... ",
            sprintf(
                "Damaging computed ignoring %s missing elementary components",
                length(missing_component)
            )
        )
	} 

	detail = list()
	
	# Room
	# Preparing ceiling
	ceiling = data_table[["room"]][c(
		"ceiling_elementary_component", 
		"ceiling_insulating_elementary_component",
		"ceiling_coating_elementary_component",
		"H_abs", "surface",	"ceiling_H")]
	ceiling[["H_abs_room"]] = ceiling[["H_abs"]]
	ceiling[["H_abs"]] = ceiling[["H_abs"]] + ceiling[["ceiling_H"]]	
	detail[["room"]] = c(
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["room"]][["floor_elementary_component"]])],
			component = split(data_table[["room"]], seq(nrow(data_table[["room"]]))),
			MoreArgs = list(
				what = "floor_elementary_component",
				dilapidation = 50, # Trouble: mix of components in this version...
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_elementary_component",
				dilapidation = 50, # Trouble: mix of components in this version...
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_insulating_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_insulating_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(ceiling[["ceiling_coating_elementary_component"]])],
			component = split(ceiling, seq(nrow(ceiling))),
			MoreArgs = list(
				what = "ceiling_coating_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE)
		)
	detail[["room"]] = detail[["room"]][!sapply(detail[["room"]], is.null)]
	
	# Furniture
	detail[["furniture"]] = c(
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["furniture"]][["component_elementary"]])],
			component = split(data_table[["furniture"]], seq(nrow(data_table[["furniture"]]))),
			MoreArgs = list(
				what = "component_elementary",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE)
		)
	detail[["furniture"]] = detail[["furniture"]][!sapply(detail[["furniture"]], is.null)]

	# Openings
	detail[["opening"]] = c(
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["opening"]][["opening_elementary_component"]])],
			component = split(data_table[["opening"]], seq(nrow(data_table[["opening"]]))),
			MoreArgs = list(
				what = "opening_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE),
		mapply(adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["opening"]][["shutter_elementary_component"]])],
			component = split(data_table[["opening"]], seq(nrow(data_table[["opening"]]))),
			MoreArgs = list(
				what = "shutter_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE)
		)
	detail[["opening"]] = detail[["opening"]][!sapply(detail[["opening"]], is.null)]
	detail[["opening"]] = lapply(detail[["opening"]], function(x){x / 2}) 
		
	# Walls
	surface_height = mapply(
        calculate_surface_height,
		wall = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
		MoreArgs = list(
            opening = data_table[["opening"]],
            version_building = version_building
        )
    )
	detail[["wall"]] = c(
		mapply(
            adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["wall_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "wall_elementary_component",
				dilapidation = 0, # Trouble : cannot do distinction between partition and load_bearing_wall...
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE
        ),
		mapply(
            adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["insulating_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "insulating_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE
        ),
		mapply(
            adjust_elementary_damaging, 
			damaging = dam[as.character(data_table[["wall"]][["render_elementary_component"]])],
			component = split(data_table[["wall"]], seq(nrow(data_table[["wall"]]))),
			surface_height = surface_height,
			MoreArgs = list(
				what = "render_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE
        )
    )
	detail[["wall"]] = detail[["wall"]][!sapply(detail[["wall"]], is.null)]
	selection = grep("partition", names(detail[["wall"]]))
	selection_names = grep("partition", names(detail[["wall"]]), value = TRUE)
	detail[["wall"]][selection] = lapply(detail[["wall"]][selection], function(x){x/2}) 
	names(detail[["wall"]][selection]) = selection_names
	
	# Coating
	surface_height = mapply(
        calculate_surface_height,
		wall = split(data_table[["coating"]], seq(nrow(data_table[["coating"]]))),
		MoreArgs = list(
            opening = data_table[["opening"]],
            version_building = version_building
        )
    )
	detail[["coating"]] = c(
		mapply(
            adjust_elementary_damaging, 
			damaging = dam[
                as.character(
                    data_table[["coating"]][["coating_elementary_component"]]
                )
            ],
			component = split(
                data_table[["coating"]],
                seq(nrow(data_table[["coating"]]))
            ),
			surface_height = surface_height,
			MoreArgs = list(
				what = "coating_elementary_component",
				dilapidation = 50,
				hazard_range = hazard_range,
				file_log = file_log),
			SIMPLIFY = FALSE
        )
    )
	detail[["coating"]] = detail[["coating"]][!sapply(detail[["coating"]], is.null)]
	
	result = list(detail = detail)
	
	# Absolute damaging
	component_category = levels(
        floodam.building::COMPONENT_ELEMENTARY[[category]]
    )
	component_category = stats::setNames(
		component_category[component_category != "building"],
		component_category[component_category != "building"]
	)
	selection = list()
	for (what in component_category) {
		selection[[what]] = sapply(
            detail[["furniture"]],
            function(x){floodam.building::COMPONENT_ELEMENTARY[attr(x, "component_elementary"), category] == what}
        )
	}
	selection[["building"]] = !apply(as.data.frame(selection), 1, any)

	building  = detail[["furniture"]][selection[["building"]]]
	building = if (length(building) == 0) detail[c("room", "opening", "wall", "coating")] else c(detail[c("room", "opening", "wall", "coating")], list(furniture = building))

	result[["absolute"]] = list(total = sumup(detail), building = sumup(building))
	for (what in component_category) {
		result[["absolute"]][[what]] = 
            Reduce(
                "+" ,
                lapply(
                    detail[["furniture"]][selection[[what]]],
                    function(x){apply(x, 1:2, sum, na.rm = TRUE)}
                )
            )
		if (is.null(result[["absolute"]][[what]])) {
            result[["absolute"]][[what]] = array(
                0,
                dim = dim(result[["absolute"]][["total"]]),
                dimnames = dimnames(result[["absolute"]][["total"]])
            )
        }
	}
	
	# Relative damaging
	result[["relative"]] = list(
        building = result[["absolute"]][["building"]] / sum(unlist(value[["monetary"]][component_category]))
    )
	for (what in component_category) {
        result[["relative"]][[what]] = result[["absolute"]][[what]] / value[[what]]
    }
	
	# Check possible problem with relative damagement
	for (what in component_category) {
		if (any(selection[[what]])) {
			value_category = sapply(result[["detail"]][["furniture"]][selection[[what]]], function(x){attr(x, "detail")[["value"]]})
			warning_what = sapply(result[["detail"]][["furniture"]][selection[[what]]], max, na.rm = TRUE) / value_category > 1
			warning_what[is.na(warning_what)] = TRUE
			if (any(warning_what)) {
				msg = c(sprintf("Possible problem with those %s components (relative damaging > 1 or value = 0):", what),
					paste("\t-", sapply(result[["detail"]][["furniture"]][selection[[what]]][warning_what], function(x){paste(attr(x, "detail")[c("component_elementary", "room_name", "H_abs")], collapse = " / ")})))
				write(msg, file_log, append = TRUE)
				warning(sprintf("Possible problem with some %s components (relative damaging > 1 or value = 0), see %s for details", what, file_log))
			}
		}
	}
	
	# Surface damaging
	result[["surface"]] = list(
		building = result[["absolute"]][["building"]] / (value[["surface"]][["internal"]])
	)
	
	if (isTRUE(verbose)) {
        message(
            sprintf(
                "\t... Damaging successfully computed for '%s'",
                model_name
            )
        )
    }
 
	invisible(result)
}


#' @title Aggregates damage from detailed list of elementary damaging functions
#' 
#' @description 
#' Aggregates detailed list of elementary damaging functions by category (total,
#' building and furniture) .
#' 
#' @param x, list of `elementary_damaging` objects
#' @param category, character indicating the model category
#' @param hazard_range list, default to
#' 	getOption("floodam_building_hazard_range")
#' 
#' @return a list of objects of class "damaging"
#'
#' @details 
#' This funtion is meant to be called during the execution of the function 
#' `compute_damage()`. It ensures that damages are calculated consistently 
#' 	across levels of aggregation : external wall segments, external walls, rooms
#' 	and building level. 
#' 
#' 	It serves as a bridge between `compute_damage()` and `sumup_damage()`, 
#' providing the latter the right list of elementary damaging components to use
#' 
#' @examples
#' \dontrun{
#' 	result[["absolute"]] = compute_damage_from_detail(
#' 		x = result[["detail"]],
#' 		category = category,
#' 		hazard_range = hazard_range
#' 	)
#' }
#' 
#' @keywords internal

compute_damage_from_detail = function(
		x, 
		category, 
		hazard_range = getOption("floodam_building_hazard_range")
	) 
{
	damage_0 = array(0, dim = lengths(hazard_range), dimnames = hazard_range)
	class(damage_0) = "damaging"

	if (is.null(x)) {
		result = list(total = damage_0, building = damage_0)
		return(result)
	}

	component_category = levels(
		as.factor(floodam.building::COMPONENT_ELEMENTARY[[category]])
	)
	component_category = stats::setNames(
		component_category[component_category != "building"],
		component_category[component_category != "building"]
	)
	selection = list()
	for (what in component_category) {
		selection[[what]] = sapply(
			x[["furniture"]],
			function(y){
				floodam.building::COMPONENT_ELEMENTARY[attr(y, "component_elementary"), category] == what
			}
		)
	}
	selection[["building"]] = !apply(as.data.frame(selection), 1, any)

	building  = x[["furniture"]][selection[["building"]]]
	building = if (length(building) == 0) {
		x[c("room", "opening", "wall", "coating")]
	} else {
		c(x[c("room", "opening", "wall", "coating")], list(furniture = building))
	}

	result = list(
		total = sumup_damaging(x),
		building = sumup_damaging(building)
	)
	for (what in component_category) {
		result[[what]] = Reduce(
			"+" ,
			lapply(
				x[["furniture"]][selection[[what]]],
				function(y){apply(y, 1:2, sum, na.rm = TRUE)}
			)
		)
	}

	# Treatment of eventual NULL slot
	selection = sapply(result, is.null)
	result[selection] = list(damage_0)

	return(result)
} 
