#' @title Perfom hydraulic analysis for a model
#'
#' @description
#' `analyse_hydraulic()` perfoms hydraulic analysis for a model.
#'
#' @param model an object of class model
#' @param limnigraph numeric, matrix of floodwater depth heights by instant t
#' 		for each exterior opening exposed to flooding
#' @param flood_duration numeric, duration of the flood event in seconds,
#' 		optional; if not provided, calculated using limnigraph
#' @param opening_scenario character, one of the following options: "open",
#' 		"close", "combine". See details.
#' @param dt_max numeric, optional; maxmimum timestep for simulation
#' @param Cd numeric, optional; discharge coefficient for each opening,
#' 		default to getOption("floodam_building_hydraulic_discharge_coefficient")
#' @param clearance numeric, distance between the opening and its
#' 		frame in centimeters. It represents water tightness for
#' 		the opening. It is a named vector of length 2, containing clearances for
#' 		`under` and the `side` of the opening.
#' 		Default to c(under = 0.5, side = 0.5)
#' @param height_break numeric, the differential height on each side of an
#' 		opening needed to break it
#' @param sim_id character, optional; id of simulation
#' @param stage character, what are the stages that should be done, default to
#' 		nothing, can be "hydraulic", "damaging", "dangerosity", "graph", "save",
#' 		"display"
#' @param what character, outputs to be saved in a temporary directory. Only
#' 		works along with stage `save`. If detail is FALSE only "hmax" and
#' 		"damage" can be saved
#' @param detail boolean, if FALSE only returns hmax and damage, else returns
#' 		all the outputs
#' @param verbose boolean, will floodam tells what it is doing, default to
#' 		getOption("floodam_building_verbose")
#' @param threshold_height numeric, height in centimeters above which a damage
#'  is considered
#' @param threshold_duration numeric, maximum duration for the damage function
#' used for interpolation. See in documentation of `get_hydraulic_damage()`.
#'
#' @details
#'
#' `stage` pilots what the function is doing:
#'
#' - "hydraulic" mandatory if you want something to be done, computes hydraulics
#' 		for the model
#' - "damaging" analyse damaging according to the hydraulics
#' - "dangerosity" analyse dangerosity according to the hydraulics
#' - "graph" saves visuals in a temporary directory
#' - "save" saves data written en parameter `what` in a temporary directory
#' - "display" displays water level if paramater `detail` is TRUE
#'
#' `opening_scenario` controls the state of openings:
#'
#' - "open": scenario where all openings are open
#' - "close": scenario where all openings are closed
#' - "combine": scenario where all exterior openings are closed and all interior
#' openings are open
#'
#' If `flood_duration` is missing, it is computed from `limnigraph` (max time +
#'  half an hour).
#'
#' @return
#'
#' list of matrix:
#'
#' - `h`: water depths in each rooms for each time step
#' - `eQ`: water exchanges through openings for each time step
#' - `v`: velocity of exchanges through openings for each time step
#' - `eS`: wet surface of exchanges through openings for each time step
#' - `hmax` : peak water depths in each rooms and their peak time
#' - `dangerosity` (optional) : dangerosity when opening a door
#' - `damage` (optional) : damage per room based on peak water depth
#'
#' @examples
#'
#' model = adu_t
#'
#' # generate limnigraph
#' flood = generate_limnigraph(
#' 	 model = model,
#'   time = c(0, 300, 900),
#'   depth = cbind(facade_1 = c(0, 3, 0)),
#'   exposition = list(
#'     facade_1 = list(external = c("wall_A", "wall_B", "wall_C", "wall_D", "wall_E",
#' 		"wall_F", "wall_G", "wall_H")))
#' )
#'
#' hydraulic = analyse_hydraulic(
#' 	model = model,
#' 	limnigraph = flood,
#' 	sim_id = "test",
#' 	stage = "hydraulic"
#' )
#'
#' @export

analyse_hydraulic = function(
		model,
		limnigraph,
		flood_duration,
		opening_scenario = c("combine", "close", "open"),
		dt_max = 0.5,
		Cd = 0.42,
		clearance = getOption("floodam_building_hydraulic_close_opening_clearance"),
		height_break = 1.0,
		sim_id = NULL,
		stage = "",
		what = c("hmax", "damage"),
		detail = TRUE,
		verbose = getOption("floodam_building_verbose"),
		threshold_height = 1,
    	threshold_duration = 12
	)
{
	start = Sys.time()
	if (identical(stage, "all")) stage = 
		getOption("floodam_building_hydrau_stage")
	stage = intersect(as.character(unlist(stage)), 
		getOption("floodam_building_hydrau_stage"))

    if (length(stage) == 0) {
        if (verbose) message("No admissible stage asked. Nothing is done.\n")
        return(NULL)
    }

	# Check clerance value and prepare permeability
	clearance = clearance[c("under", "side")]
	if (length(clearance) == 0) {
		if (verbose) message("Need at list one clearances 'under' or 'side' for the opening.\n")
        return(NULL)
	}

	permeability = switch(
		length(clearance),
		names(clearance),
		"both"
	)

	# Perform hydraulic analysis
	if ("hydraulic" %in% stage) {
		# initial checks
		opening_scenario = match.arg(opening_scenario)
		if (missing(flood_duration)) {
			flood_duration = 2 * 
				max(limnigraph[["limnigraph"]][, "time"]) + 1800
		}

		if (verbose) {
			message(
				sprintf("Simulating hydraulics for '%s'...\n", model[["name"]]),
				appendLF = FALSE
			)
		}
		
		# determine opening scenario
		data_hyd = prepare_hydraulic_input(
			model,
			exposition = limnigraph[["exposition"]],
			clearance = clearance,
			permeability = permeability
		)
		opening = data_hyd[[opening_scenario]]
		open = data_hyd[["open"]]

		# adding informations for the simulation
		opening[["discharge_coeff"]] = Cd
		opening[["H_top"]] = opening[["H_abs"]] + opening[["height"]]
		room = data_hyd[["room"]]
		room[["room"]] = as.character(room[["room"]])
		room[["v"]] = room[["surface"]] * room[["ceiling_H"]] / 100

		r_names_to_merge = NULL
		# in case of invisible walls, merge rooms
		if("empty" %in% model[["data_table"]][["wall"]][["type"]])
		{
			r_names_to_merge = find_room_to_merge(model, verbose)
		}

		# starting the simulation
		result = compute_hydraulic(
			opening = opening,
			room = room,
			bound = limnigraph[["limnigraph"]],
			open = open,
			dt_max = dt_max,
			t_start = 0,
			t_max = flood_duration,
			height_break = height_break,
			r_names_to_merge = r_names_to_merge,
			detail = detail
		)

		# Can be written more efficiently
		# Sum of flows when flows can go by the bottom and the side of closed
		# openings
		if (permeability == "both" && isTRUE(detail))
		{
			colname_base = sub("_v", "", colnames(result[["eQ"]]))
			op = unique(colname_base)
			summed_res = list()

			for (n in op)
			{
				match_col = which(colname_base == n)
				summed_res[[n]] = rowSums(result[["eQ"]][, match_col, drop = FALSE])
			}

			result[["eQ"]] = do.call(cbind, summed_res)
		}

		class(result) = "hydraulic"

		if (verbose) {
			message(
				sprintf(
					"\t... hydraulics successfully modeled for '%s'",
					model[["name"]]
				)
			)
		}
	}

	# compute dangerosity
	if ("dangerosity" %in% stage) {
		opening = data_hyd[["open"]]

		# adding informations for the simulation
		opening[["discharge_coeff"]] = Cd
		opening[["H_top"]] = opening[["H_abs"]] + opening[["height"]]
		result[["danger"]] = compute_danger(
			result_h = result[["h"]],
			opening = opening,
			room = room,
			bound = limnigraph[["limnigraph"]],
			dt_max = dt_max,
			t_max = flood_duration)

		message(
			sprintf(
				"\t... dangerosity successfully computed for '%s'",
				model[["name"]]
			)
		)
	}

	# compute damage
	if ("damaging" %in% stage) {
		if (is.null(model[["dam_room_wall"]][["absolute_room"]])) {
			stop("You need to enable parameter `hydraulic` in analyse_model()")
		}
		result[["damage"]] = get_hydraulic_damage(
			result[["hmax"]]["z", ],
			model[["dam_room_wall"]][c("absolute_room", "absolute_external")],
			limnigraph[["exposition"]],
			threshold_height = threshold_height,
			threshold_duration = threshold_duration
		)

		if (verbose) {
			message(
				sprintf(
					"\t... damaging successfully computed for '%s'",
					model[["name"]]
				)
			)
		}
	}

	# Saving outputs
	if ("save" %in% stage) {
		name = sprintf(
			"%s-%s-%s",
			model[['name']],
			opening_scenario,
			if (is.null(sim_id)) "NA" else sim_id
		)
		path = model[["path"]][["model_output_hydraulic"]]
		for (what in intersect(names(result), what)) {
			data.table::fwrite(
				as.data.frame(result[[what]]),
				file.path(path, sprintf("%s-%s.csv.gz", name, what))
			)
		}
		message(
			sprintf(
				"\t... hydraulics successfully saved for '%s' in '%s'",
				model[["name"]],
				path
			)
		)
	}

	# Plotting graph damaging
	if ("graph" %in% stage && detail) {
		if (verbose) message(sprintf("Plotting graphs for '%s'...", model[["name"]]))
		selected_view = c("height", "discharge")
		msg_plot = plot(result,
			output_dir = model[["path"]][["model_output_hydraulic"]],
			view = selected_view,
			device = "png")
		if (verbose) message(paste("\t-", msg_plot))
	}

	# Plotting something in display
	if ("display" %in% stage) {
		if(isFALSE(detail)) {
			message(
				sprintf(
					"\t... can't display graphs when paramater `detail` is FALSE"
				)
			)
		} else {
			if (verbose) message(sprintf("Display height view of '%s'", sim_id))
			plot(result,
				output_dir = model[["path"]][["model_output_hydraulic"]],
				view = "height",
				device = "display"
			)
		}
	}

	duration = Sys.time() - start

	if (verbose) {
		message(
			sprintf(
				"End of analysis for '%s'. Total elapsed time %.2f %s\n",
				model[['name']],
				duration,
				attr(duration, "units")
			)
		)
	}

	invisible(result)
}
