#' Adjust model information in the table furniture
#' 
#' @param data_table list of table informations of the model
#' @param dilapidation, numeric, dilapidation rate to be applied, default to 0
#' @param category list of table informations of the model
#' @param file_log path of the file where to write log information
#' @return the furniture table adjusted
#' 
#' @keywords internal

adjust_furniture_03 = function(data_table, dilapidation, category, file_log) {
	
	if (is.null(data_table[["furniture"]])) return(NULL)
	
	# Remove all fields that are supposed to be updated (necessary when adjusting an existing model)
	data_table[["furniture"]][["storey"]] = data_table[["furniture"]][["H_abs"]] = data_table[["furniture"]][["value"]] = NULL
	for (i in grep("category", names(data_table[["furniture"]]), value = TRUE)) data_table[["furniture"]][[i]] = NULL
	
	# Adjusting H_abs	
	result = keep_df_order(data_table[["furniture"]], merge, data_table[["room"]][c("building", "storey", "room", "H_abs")], by.x = "room_name", by.y = "room")
	selection = grep("^H_abs$|^relative_height$", names(result))
	result[["H_abs"]] = apply(result[selection], 1, sum, na.rm = TRUE)
	
	# Adjusting dilapidation
	result[["dilapidation"]][is.na(result[["dilapidation"]])] = dilapidation

	# Adjusting value	
	default = floodam.building::COMPONENT_ELEMENTARY
	default[["component_elementary"]] = row.names(default)
	default = default[c("component_elementary", "v_min", "v_max", category)]
	
	result = merge(result, default, by = "component_elementary", all.x = TRUE)
	result[["v_min"]] = result[["v_min.x"]]
	result[["v_min"]][is.na(result[["v_min"]])] = result[["v_min.y"]][is.na(result[["v_min"]])]
	result[["v_max"]] = result[["v_max.x"]]
	result[["v_max"]][is.na(result[["v_max"]])] = result[["v_max.y"]][is.na(result[["v_max"]])]
	
	result[["v_min.x"]] = result[["v_min.y"]] = result[["v_max.x"]] =result[["v_max.y"]] = NULL
	
	result[["value"]] = apply(result[c("v_min", "v_max")], 1, mean, na.rm = TRUE) * (1 - result[["dilapidation"]] / 100) * result[["quantity"]]

	# Some tests on problems that shall occur, warning emission and log update_
	if (any(is.nan(result[["value"]]))) {		
		report = "\nFollowing furniture do not value! Please check. O taken as default value...\n"
		report = paste(report, paste("\t- ", unique(as.character(result[["component_elementary"]][is.nan(result[["value"]])])), "\n", sep = "", collapse = ""))
		write(report, file = file_log, append = TRUE)
		warning(sprintf("Some furniture do not have value! O taken as default value... Look at %s", file_log))
		result[["value"]][is.nan(result[["value"]])] = 0
	}
	
	invisible(result)
}

#' Adjust model information in the table furniture
#' 
#' @param data_table list of table informations of the model
#' @param dilapidation, numeric, dilapidation rate to be applied, default to 0
#' @param category list of table informations of the model
#' @param file_log path of the file where to write log information
#' @return the furniture table adjusted
#' 
#' @keywords internal

adjust_furniture_02 = function(data_table, dilapidation, category, file_log) {
	
	if (is.null(data_table[["furniture"]])) return(NULL)
	
	# Remove all fields that are supposed to be updated (necessary when adjusting an existing model)
	data_table[["furniture"]][["storey"]] = data_table[["furniture"]][["H_abs"]] = data_table[["furniture"]][["value"]] = NULL
	for (i in grep("category", names(data_table[["furniture"]]), value = TRUE)) data_table[["furniture"]][[i]] = NULL
	
	# Adjusting H_abs	
	result = keep_df_order(
		data_table[["furniture"]],
		merge,
		data_table[["room"]][c("storey", "room", "H_abs")],
		by.x = "room_name",
		by.y = "room"
	)
	selection = grep("^H_abs$|^relative_height$", names(result))
	result[["H_abs"]] = apply(result[selection], 1, sum, na.rm = TRUE)
	
	# Adjusting dilapidation
	result[["dilapidation"]][is.na(result[["dilapidation"]])] = dilapidation

	# Adjusting value	
	default = floodam.building::COMPONENT_ELEMENTARY
	default[["component_elementary"]] = row.names(default)
	default = default[c("component_elementary", "v_min", "v_max", category)]
	
	result = merge(result, default, by = "component_elementary", all.x = TRUE)
	result[["v_min"]] = result[["v_min.x"]]
	result[["v_min"]][is.na(result[["v_min"]])] = result[["v_min.y"]][is.na(result[["v_min"]])]
	result[["v_max"]] = result[["v_max.x"]]
	result[["v_max"]][is.na(result[["v_max"]])] = result[["v_max.y"]][is.na(result[["v_max"]])]
	
	result[["v_min.x"]] = result[["v_min.y"]] = result[["v_max.x"]] =result[["v_max.y"]] = NULL
	
	result[["value"]] = apply(result[c("v_min", "v_max")], 1, mean, na.rm = TRUE) * (1 - result[["dilapidation"]] / 100) * result[["quantity"]]

	for (column in category) {
		result[[column]] = factor(result[[column]], levels = unique(default[[category]]))
	}
	
	# Some tests on problems that shall occur, warning emission and log update.
	if (any(is.nan(result[["value"]]))) {		
		report = sprintf(
			"Following furniture do not value! Please check. O taken as default value...\n\t- %s\n",
			unique(as.character(result[["component_elementary"]][is.nan(result[["value"]])]))
		)
		write(
			report,
			file = file_log,
			append = TRUE
		)
		warning(
			sprintf(
				"Some furniture do not have value! O taken as default value... Look at %s",
				file_log
			)
		)
		result[["value"]][is.nan(result[["value"]])] = 0
	}
	
	invisible(result)
}
	
