## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk[["set"]](
  collapse = TRUE,
  comment = "#>",
  fig.align = "center"
)

## ----setup--------------------------------------------------------------------
#> Loading required package: floodam.building
library(floodam.building)

#> set up model to use example shipped with floodam
model_path = list(
data = system.file("extdata", package = "floodam.building"),
output = tempdir()
)

## ----model, message = FALSE---------------------------------------------------
model = analyse_model(
  model = "adu_t",
  type = "adu",
  stage = c("load", "extract", "hydraulic"),
  verbose = FALSE,
  path = model_path
)

## ----modhyd-------------------------------------------------------------------
head(model[["hydraulic"]][c("exchanges_close", "rooms")], 10) |> knitr::kable()

## ----limnigraph---------------------------------------------------------------
flood = generate_limnigraph(
  model = model,
  time = c(0, 5400, 10800),
  depth = cbind(facade_1 = c(0, 2, 0)),
  exposition = list(
    facade_1 = list( external = c("wall_A", "wall_B", "wall_C", "wall_D",
      "wall_E", "wall_F", "wall_G", "wall_H")))
)

## ----hydraulic----------------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic"),
  sim_id = "integrated_model"
)

## ----hydraulic_output---------------------------------------------------------
head(hydraulic[["h"]], 10) |> knitr::kable()

## ----depth, fig.height=5.6, fig.width=7.1-------------------------------------
#> visualization of the floodwater depth
plot(hydraulic, view = "height")

## ----discharge, fig.height=5.6, fig.width=7.1---------------------------------
#> visualization of the floodwater discharge in the openings
plot(hydraulic, view = "discharge")

## ----openings-----------------------------------------------------------------
head(model[["data_table"]][["opening"]][c("room", "name")], 10) |> knitr::kable()

## ----hydraulic_save-----------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic", "graph", "save"),
  what = c("h", "eQ"),
  sim_id = "integrated_model"
)

## ----pdf, fig.height=5.6, fig.width=7.1---------------------------------------
#> visualization of the floodwater discharge in the openings
plot(hydraulic, view = "height", device = "pdf", output_path = model[["path"]][["model_output_hydraulic"]])

## ----model_basment, message = FALSE-------------------------------------------
model_basement = analyse_model(
  model = "adu_t_basement",
  type = "adu",
  stage = c("load", "extract", "hydraulic"),
  verbose = FALSE,
  path = model_path
)

## ----limnigraph_basment-------------------------------------------------------
# flood = generate_limnigraph(
#   time = c(0, 5400, 10800),
#   depth = cbind(external_1 = c(0, 2, 0)),
#   external = list(
#     external_1 = list(groundfloor = c("wall_A", "wall_B", "wall_C", "wall_D", "wall_E", "wall_F",
#       "wall_G", "wall_H"),
#       basement = c("wall_A", "wall_B", "wall_C", "wall_D")))
# )

## ----hydraulic_basement-------------------------------------------------------
hydraulic = analyse_hydraulic(
  model = model,
  limnigraph = flood,
  opening_scenario = "close",
  stage = c("hydraulic"),
  sim_id = "integrated_model"
)

## ----depth_basement, fig.height=5.6, fig.width=7.1----------------------------
#> visualization of the floodwater depth
plot(hydraulic, view = "height")

