#' Update models of a given type.
#' 
#' @param type character, type of model that is loaded, default to NULL
#' @param hazard_range if damaging are calculated, gives the range for hazard, default to getOption("floodam_building_hazard_range")
#' @param path list of useful paths, default to getOption("floodam_building_path")
#' @param graph boolean, will graphs be plotted, default to getOption("floodam_building_graph")
#' @param verbose boolean, will floodam tells what it is doing, default to getOption("floodam_building_verbose")
#' @param sel character, models to be selected within type, default to NULL (everything is selected)
#' @param skip character, models not to be selected within types, default to NULL (nothing is skipped)
#' @return a list of models
#' @examples
#' \dontrun{
#' update_type("activity")
#' }
#' 
#' @export

update_type = function(
		type,
		hazard_range = getOption("floodam_building_hazard_range"), 
		path = getOption("floodam_building_path"),
		graph = getOption("floodam_building_graph"),
		verbose = getOption("floodam_building_verbose"),
		sel = NULL,
		skip = NULL
	)
{
	
	start = Sys.time()
		
	path = update_path(type = type, path = path)
	selection = dir(path[["model"]])
	if (!is.null(sel)) {
		missing = sel[!(sel %in% selection)]
		selection =  sel[sel %in% selection]
		if (length(missing) > 0) {
			warning(
				sprintf(
					"Those models (%s) are not in %s. Not evaluated.",
					paste(missing, collapse = ", "),
					path[["model"]]
				)
			)
		}
	} else if (!is.null(skip)) {
		selection = selection[!(selection %in% skip)]
	}

	if (length(selection) == 0) {
		warning(sprintf("No model found in %s", path[["model"]]))
		return(NULL)
	}
	
	if (length(grep("parallel", search())) > 0) {
		model = parallel::mclapply(
			selection,
			analyse_model,
			type = type,
			stage = "all",
			verbose = FALSE,
			mc.cores = parallel::detectCores()
		)
	} else {
		model = lapply(
			selection,
			analyse_model,
			type = type,
			stage = "all",
			verbose = FALSE
		)
	}
	names(model) = selection

	# Neccesary but so beurk !	
	model_values = t(sapply(model, function(x){x[["value"]]}))
	cn = names(model_values[1,])
	rn = row.names(model_values)
	model_values = matrix(
		unlist(model_values),
		nrow = length(rn),
		ncol = length(cn),
		dimnames = list(rn, cn)
	)
	utils::write.csv2(model_values, file.path(path[["model_output"]], "model-values.csv"))
	
	duration = Sys.time() - start
	if (verbose) {
		message(
			sprintf(
				"End of analysis for %s model. Total elapsed time %.2f %s\n",
				length(model),
				duration,
				attr(duration, "units")
			)
		)
	}
	
	invisible(model)
}
