#' Check whether the global variable LEGEND is up to date and update it if necessary
#'
#' @param legend_table data.frame, default to floodam.building::LEGEND
#' @param component_elementary character
#' @return updated data.frame, removing component_elementary that should not be present, adding those that should be (random selection of colors)
#' 
#' @keywords internal

update_legend = function(
		legend_table = floodam.building::LEGEND,
		component_elementary = names(floodam.building::DAM)
	)
{

	# Check that every item in legend is in DAM (the only elements not in DAM should correspond to a wall_type or opening_type)
	wall_type = c("load_bearing_wall", "lining", "partition", "empty")
	opening_type = c("door", "window", "french_window")
	not_in_dam = which(!(row.names(legend_table) %in% c(component_elementary, wall_type, opening_type))) 

	if (length(not_in_dam) > 0) {
		message(sprintf("Following elements in LEGEND are obsolete and removed from LEGEND.csv: %s...\n", 
			paste(row.names(legend_table[not_in_dam,]), collapse = ", ")))
		legend_in_dam = legend_table[-not_in_dam,]
		utils::write.csv2(legend_in_dam, "data-floodam/parameter/LEGEND.csv", row.names = TRUE)
	}

	# Check that every item containing "floor" or "wall_coating" in DAM is in legend_in_dam
	dam_no_legend = component_elementary[!(component_elementary %in% row.names(legend_table))]
	floor = grep("floor", dam_no_legend, value = TRUE)
	wall_coating = grep("wall_coating", dam_no_legend, value = TRUE)

	if (length(c(floor, wall_coating)) == 0) {
		message("LEGEND is up-to-date.\n")
	} else {
	# if not every item in DAM is in legend_in_dam: 
		message(sprintf("The following elements are missing : %s...\n", paste(c(floor, wall_coating), collapse = ", ")))

		if (length(floor) > 0) {
			message(sprintf("Adding missing floor elements to LEGEND...\n"))
			legend_in_dam = add_component(legend_in_dam, floor, lwd = 1, density = -1)
		}

		if (length(wall_coating) > 0){
			message(sprintf("Adding missing wall_coating elements to LEGEND...\n"))
			legend_in_dam = add_component(legend_in_dam, wall_coating, lwd = 1, density = -1)  
		}

		utils::write.csv2(legend_in_dam, "data-floodam/parameter/LEGEND.csv", row.names = TRUE)
	}
	
	# return(legend_in_dam)
}

add_component = function(
		legend_in_dam, 
		item, 
		lwd = NA, 
		density = NA
	)
{
	col = sample(grDevices::colors()[!(grDevices::colors() %in% legend_in_dam[["col"]])], size = length(item), replace = FALSE)
	lwd = rep(lwd, length(item))
	density = rep(density, length(item))
	legend_in_dam = rbind(legend_in_dam, data.frame(row.names = item, "col" = col, "lwd" = lwd, "density" = density))

	return(legend_in_dam)
}