#' Summarize detailed list of elementary damaging functions.
#' 
#' @param x, object of class elementary_damaging or a list
#' @param who, NULL or character to trace the name of the calling function if an error occurs
#' @return an object of class "damaging"
#' @examples
#' 
#' 	model = analyse_model(model = adu_t, stage = c("extract", "damaging"))
#' 	damage = sumup_damaging(model[["damaging"]][["detail"]])
#' 
#' 
#'  model = analyse_model(adu_t_basement, stage = c("extract", "damaging"))
#'  selection = grepl(
#'  	"external",
#'  	levels(model[["data_table"]][["wall"]][["room"]])
#'  )
#'  
#'  external = split(
#'  	as.character(model[["data_table"]][["wall"]][["wall"]]), 
#'  	model[["data_table"]][["wall"]][["room"]]
#'  )[selection]
#'  
#'  detail_by_room = extract_damage_detail_by_room(
#'  	x = model[["damaging"]][["detail"]], 
#'  	room = levels(model[["data_table"]][["room"]][["room"]])
#'  )
#'  
#'  damage = sumup_damaging(detail_by_room[["external_basement"]])
#' 
#' @export

sumup_damaging = function(x, who = NULL) {
	if (is.null(who)) who = deparse(sys.call())
	if (methods::is(x, "elementary_damaging")) {
        return(apply(x, 1:2, sum, na.rm = TRUE))
    }
	if (is.list(x)) {
		x = x[!sapply(x, is.null)]
		if (length(x) == 0) {
			return(NULL)
		}
		result = Reduce("+", lapply(x, sumup_damaging, who = who))
		class(result) = "damaging"
		return(invisible(result))
	}
	stop(
		sprintf(
			"In %s, something inside is not an elementary_damaging found (%s). Please check.",
			who,
			class(x)),
			call. = FALSE
	)
}

