#' Save a model
#' 
#' @param model model
#' @param verbose logical, default to getOption("floodam_building_verbose")
#' @return An object of class "model" with those characteristics
#' \describe{
#'   \item{building}{list, representation of information stored in the building file}
#'   \item{path}{list, useful paths to read and write data}
#'   \item{file_name}{list, useful file names to read and write data}
#' }#
#' @examples
#' 
#' 	model_path = list(
#' 		data = system.file("extdata", package = "floodam.building"),
#' 		output = tempdir()
#' 	)
#' 	model = analyse_model(
#' 		model = "adu_t", 
#' 		type = "adu", 
#' 		stage = c("load", "extract", "damaging"), 
#' 		path = model_path
#' 	)
#' 	save_model(model)
#' 
#' @export

save_model = function(
		model,
		verbose = getOption("floodam_building_verbose")
	)
{
	if (verbose) message(sprintf("Saving informations for '%s'...\n", model[["name"]]))
	
	# building as xml
	XML::saveXML(
		list2xml(model[["building"]]), 
		file.path(
			model[["path"]][["model_output_damaging"]],
			"building.xml"
		), 
		encoding = "UTF-8", 
		prefix = '<?xml version="1.0" encoding="UTF-8"?>\n'
	)
	if (verbose) message("\t- building successesfully saved as xml\n")	

	# building as yaml
	yaml::write_yaml(
		model[["building"]],
		file.path(
			model[["path"]][["model_output_damaging"]],
			"building.yaml"
		)
	)
	if (verbose) message("\t- building successesfully saved as yaml\n")	

	# data_table
	for (element in names(model[["data_table"]])) {
		if (element != "parameter") {
			utils::write.csv2(
				model[["data_table"]][[element]], 
				file.path(
					model[["path"]][["model_output_damaging"]],
					sprintf("data_table_%s.csv", element)
				),
				row.names = FALSE
			)
		}
	}
	if (verbose) message("\t- all 'data_table' successesfully saved as csv\n")	

	# data_table
	for (element in names(model[["data_table"]][["parameter"]])) {
		utils::write.csv2(
			model[["data_table"]][["parameter"]][[element]], 
			file.path(
				model[["path"]][["model_output_damaging"]],
				sprintf("parameter_%s.csv", element)
			),
			row.names = FALSE
		)
	}
	if (verbose) message("\t- all 'parameter' successesfully saved as csv\n")	

	# value
	for (element in names(model[["value"]])) {
		utils::write.csv2(
			model[["value"]][[element]], 
			file.path(
				model[["path"]][["model_output_damaging"]],
				sprintf("value_%s.csv", element)
			),
			row.names = FALSE
		)
	}
	if (verbose) message("\t- all 'value' successesfully saved as csv\n")

	# damaging
	utils::write.csv2(
		summarize_model_damaging(model), 
		file.path(
			model[["path"]][["model_output_damaging"]],
			model[["file_name"]][["output_damaging.csv"]]
		), 
		row.names = FALSE
	)
	if (verbose) message("\t- summarized 'damaging' successesfully saved as csv\n")

	if (verbose) message(sprintf("\t... Informations successfully saved for '%s'\n", model[["name"]]))
}
