#' Plot relative damaging function
#' 
#' @param damaging array giving damaging function
#' @param d_select selected values of durations where to make some focus. At list one value.
#' @param h_range, height range for making the graphs. Default to NULL, and so analyzing dam
#' @param d_range, duration range for making the graphs. Default to NULL, and so analyzing dam
#' @param name, name of the elementary component processed. Default to NULL, no title.
#' @param output path for saving the result. Default to NULL, nothing is saved, graph is displayed.
#' @param compare a list of relative damaging functions to compare damaging with.
#' @return NULL
#' 
#' @keywords internal

plot_relative_damaging = function(
		damaging,
		d_select = getOption("floodam_building_d_select"),
		h_range = NULL,
		d_range = NULL,
		name = NULL,
		output = NULL,
		compare = NULL
	)
{	
  
	graphics::close.screen(all = TRUE)
	if (!is.null(output)) 
	{
		grDevices::pdf(file.path(output, paste(name, ".pdf", sep = "")), width = 15 , height = 10)
	}

	# Prepare 4 different screens for general layout
	mat = rbind(c(0, 1, .9, 1), c(0, 1, .55, .9), c(0, 1, .2, .55), c(0, 1, 0, .2))
	graphics::split.screen(mat)
	  
	graphics::screen(1)
	if (!is.null(name)) {
		graphics::title(gsub("[_]", " ", name), cex.main = 1.5)
	}

	# Deal with general graphic, prepare 3 subscreens in screen 2
	mat = rbind(c(0, .45, 0, 1), c(0.45, .5, 0, 1), c(.5, 1, 0, 1))
	graphics::split.screen(mat, screen = 2)

	# Plot the total damage as an image
	graphics::screen(5)
	xlim = if (is.null(h_range)) range(as.numeric(rownames(damaging[["mean"]]))) else h_range
	ylim = if (is.null(d_range)) range(as.numeric(colnames(damaging[["mean"]]))) else d_range  
	suppressWarnings(
		fields::image.plot(
			x = as.integer(rownames(damaging[["mean"]])), 
			y = as.integer(colnames(damaging[["mean"]])), 
			z = damaging[["mean"]], 
			col = grDevices::colorRampPalette(c("blue", "red"))(10),  #gray((10:0)/10), 
			xlab = "h (cm)", ylab = "d (hours)",
			xlim = xlim, ylim = ylim,
			zlim = c(0, max(1, max(damaging[["mean"]], na.rm = TRUE))),
			main = "Full space definition"
  		)
	)

	# Plot the mean damage, one line by duration
	graphics::screen(7)
	plot(
		NA,
		xlim = range(as.integer(rownames(damaging[["mean"]]))), 
		ylim = range(0, max(1, max(damaging[["mean"]], na.rm = TRUE))), 
		xlab = "h (cm)", ylab = "damaging",
		main = "Total"
	)
	col = grDevices::rainbow(length(d_select))
	lapply(as.character(d_select), 
		function(z) {
		graphics::lines(as.integer(rownames(damaging[["mean"]])), damaging[["mean"]][ , z], 
				type = "l", col = col[which(d_select == z)])
		}
	)
	graphics::legend("topleft", title = "Duration", legend = paste(d_select, "hours"), lwd = 1, seg.len = .5, col = col)
	graphics::close.screen(5:7)
	
	# Plot damage by sub.curve, one plot by duration
	graphics::split.screen(c(1, length(d_select)), screen = 3)
	col = grDevices::rainbow(length(damaging), start = .85, end = .15)
	names(col) = names(damaging)
	if (!is.null(compare)) {
		col_compare = grDevices::rainbow(length(compare), start = .3, end = .7)
		names(col_compare) = names(compare)
	}
	for (i in seq(length(d_select))) {
		graphics::screen(i + 4)
		plot(NA,
			xlim = range(as.integer(rownames(damaging[["mean"]]))), 
			ylim = range(0, max(1, max(unlist(damaging), na.rm = TRUE))), 
			xlab = "h (cm)", ylab = "damaging",
			main = sprintf("Duration: %s hours", d_select[i])
		)
		
		for (a in names(damaging)) {
			graphics::lines(as.integer(rownames(damaging[["mean"]])), damaging[[a]][ , as.character(d_select[i])], type = "l", col = col[a])
		}
		if (!is.null(compare)) {
			for (a in names(compare)) {
				graphics::lines(as.integer(rownames(damaging[["mean"]])), compare[[a]][ , as.character(d_select[i])], type = "l", col = col_compare[a])
			}
		}
	}
	graphics::close.screen(4 + seq(length(d_select)))
	
	# Plot legend of damage by actions, one legend for all
	mat = rbind(c(0, 1, 0.5, 1), c(0, 1, 0, 0.5))
	graphics::split.screen(mat, screen = 4)
	graphics::screen(5)
	if (is.null(compare)) {
		graphics::legend("center", legend = names(damaging), lwd = 1, col = col, horiz = TRUE)
	} else {
		graphics::legend("center", legend = names(damaging), lwd = 1, col = col, horiz = TRUE)
		graphics::screen(6)
		graphics::legend("center", legend = names(compare), lwd = 1, col = col_compare, horiz = TRUE)
	}
	
	graphics::close.screen(all = TRUE)
	if (!is.null(output)) grDevices::dev.off()
  
	invisible(NULL)
}
