#' plot hydraulics.
#' 
#' @param x an object of class hydraulic
#' @param output_dir character, path directory where output is saved. Default to
#' 	current directory
#' @param device, character, defines way of plotting. Can only be one device at
#' 	a time. If `display` is selected only one view can be shown at a time
#' @param view, vector of character, defines which view shall be plotted
#' 	Three options: "height", "discharge", "level"
#' @param selection vector, select the objects you want to plot either rooms or
#' facades or openings
#' @param ... some extra parameters (unused)
#' 
#' @details
#' 
#' This is a new method for the plot function. It can be used to display graphs
#' or save graphs in different formats (`.png` and `.pdf`). The `.png` is in a 
#' lower resolution than `.pdf` but is much lighter.
#' 
#' 3 differents views are proposed:
#' 
#' - **height**: it represents the evolution of water height (m) for each room, the
#' reference being the elevation of each room.
#' - **level**: it represents the evolution of water height (m) for each room, the
#' reference being the elevation of the whole model.
#' - **discharge**: it represents the evolution of discharge (m³/s) for each
#' opening.
#' 
#' @return Nothing
#' 
#' @examples
#' 
#' # declaring input and output paths
#' model = adu_t
#' 
#' # generate limnigraph
#' flood = generate_limnigraph(
#' 	 model = model,
#'   time = c(0, 300, 900),
#'   depth = cbind(facade = c(0, 3, 0)),
#'   exposition = list(
#'     facade = list(external = c("wall_A", "wall_B", "wall_C", "wall_D", "wall_E",
#' 		"wall_F", "wall_G", "wall_H")))
#' )
#' 
#' hydraulic = analyse_hydraulic(
#' 	model = model,
#' 	limnigraph = flood,
#' 	opening_scenario = "close",
#' 	sim_id = "test",
#' 	stage = "hydraulic"
#' )
#' 
#' plot(hydraulic)
#' plot(hydraulic, device = "png", view = c("height", "discharge"), output_dir = tempdir())
#'  
#' @export

plot.hydraulic = function(
		x,
		output_dir = ".",
		device = "display",
		view = c("height", "discharge"),
		selection = "all",
		...
	)
{
	plot_hydraulic(x, output_dir, device, view, selection, ...)
}

plot_hydraulic = function(
		x,
		output_dir,
		device = c("display", "pdf", "png"),
		view = c("height", "discharge", "level"),
		selection = "all",
		...
	)
{
	# Some checks on option
	device = match.arg(device)
	view = match.arg(view, several.ok = TRUE)

	if (device == "display") view = view[[1]]

	msg = c()
	if ("height" %in% view) {
		msg = c(
			msg, 
			plot_hydraulic_height(
				x = x,
				output_dir = output_dir,
				device = device,
				selection = selection,
				...
			)
		)
	}
	if ("discharge" %in% view) {
		msg = c(
			msg, 
			plot_hydraulic_discharge(
				x = x,
				output_dir = output_dir,
				device = device,
				selection = selection,
				...
			)
		)
	}
	if ("level" %in% view) {
		msg = c(
			msg, 
			plot_hydraulic_level(
				x = x,
				output_dir = output_dir,
				device = device,
				selection = selection,
				...
			)
		)
	}
	
	if (device != "display") return(msg)
}	

plot_hydraulic_height = function(
	x,
	output_dir,
	device,
	selection = "all",
	...
) {
	
	plot_height = function(x) {

		if (identical(selection, "all")) selection = colnames(x[["h"]])[-1]

		cols = grDevices::rainbow(length(selection))
		plot(
			1,
			type = "n",
			xlim = c(0, max(x[["h"]][, "time"] / 3600)),
			ylim = c(0, max(x[["hmax"]]["z", selection])),
			xlab = "time [h]",
			ylab = "height [m]",
			main = "Water height in rooms"
		)
		id = 1
		for (room in selection) {
			graphics::lines(x[["h"]][, "time"] / 3600, x[["h"]][, room], col = cols[id])
			id = id + 1
		}
		graphics::legend("topright", legend = selection,
			col = cols, lwd = 2.0,
			title = "Room")
		graphics::grid()
	}

	# Preparing
	graphics::close.screen(all.screens = TRUE)

	if ("display" %in% device) {
		default_par = graphics::par(no.readonly = TRUE)
		plot_height(x)
		graphics::par(default_par)
	}

	if ("pdf" %in% device) {
		grDevices::pdf(file.path(output_dir, "water_height.pdf"))
		on.exit(grDevices::dev.off())
		plot_height(x)
		msg = sprintf(
			"Water height view save in %s",
			file.path(output_dir, "water_height.pdf")
		)
		return(msg)
	}

	if ("png" %in% device) {
		grDevices::png(file.path(output_dir, "water_height.png"))
		on.exit(grDevices::dev.off())
		graphics::plot.new()
		plot_height(x)
		msg = sprintf(
			"Water height view save in %s",
			file.path(output_dir, "water_height.png")
		)
		return(msg)
	}
}

plot_hydraulic_discharge = function(
	x,
	output_dir,
	device,
	selection = "all",
	...
) {

	plot_discharge = function(x) {

		if (identical(selection, "all")) selection = colnames(x[["eQ"]])[-1]

		cols = grDevices::rainbow(length(selection))
		plot(
			1,
			type = "n",
			xlim = c(0, max(x[["eQ"]][, "time"] / 3600)),
			ylim = c(min(x[["eQ"]][, selection]), max(x[["eQ"]][, selection])),
			xlab = "time [h]",
			ylab = "discharge [m3/s]",
			main = "Discharge in the openings"
		)
		id = 1
		for (opening in selection) {
			graphics::lines(x[["eQ"]][, "time"] / 3600, x[["eQ"]][, opening],
				col = cols[id])
			id = id + 1
		}
		graphics::legend("topright", legend = selection,
			col = cols, lwd = 2.0, 
			title = "Opening")
		graphics::grid()
	}
	# Preparing
	graphics::close.screen(all.screens = TRUE)

	if ("display" %in% device) {
		default_par = graphics::par(no.readonly = TRUE)
		plot_discharge(x)
		graphics::par(default_par)
	}

	if ("pdf" %in% device) {
		grDevices::pdf(file.path(output_dir, "water_discharge.pdf"))
		on.exit(grDevices::dev.off())
		plot_discharge(x)
		msg = sprintf(
			"Water height view save in %s",
			file.path(output_dir, "water_discharge.pdf")
		)
		return(msg)
	}

	if ("png" %in% device) {
		grDevices::png(file.path(output_dir, "water_discharge.png"))
		on.exit(grDevices::dev.off())
		graphics::plot.new()
		plot_discharge(x)
		msg = sprintf(
			"Water height view save in %s",
			file.path(output_dir, "water_discharge.png")
		)
		return(msg)
	}
}

plot_hydraulic_level = function(
	x,
	output_dir,
	device,
	selection = "all",
	...
) {
	
	plot_level = function(x) {

		if (identical(selection, "all")) selection = colnames(x[["z"]])[-1]
		
		cols = grDevices::rainbow(length(selection))
		plot(
			1,
			type = "n",
			xlim = c(0, max(x[["z"]][, "time"] / 3600)),
			ylim = c(min(x[["z"]][, selection]), max(x[["z"]][, selection])),
			xlab = "time [h]",
			ylab = "level [m]",
			main = "Water level in rooms"
		)
		id = 1
		for (room in selection) {
			graphics::lines(x[["z"]][, "time"] / 3600, x[["z"]][, room], col = cols[id])
			id = id + 1
		}
		graphics::legend("topright", legend = selection,
			col = cols, lwd = 2.0,
			title = "Room")
		graphics::grid()
	}

	# Preparing
	graphics::close.screen(all.screens = TRUE)

	if ("display" %in% device) {
		default_par = graphics::par(no.readonly = TRUE)
		plot_level(x)
		graphics::par(default_par)
	}

	if ("pdf" %in% device) {
		grDevices::pdf(file.path(output_dir, "water_level.pdf"))
		on.exit(grDevices::dev.off())
		plot_level(x)
		msg = sprintf(
			"Water level view save in %s",
			file.path(output_dir, "water_level.pdf")
		)
		return(msg)
	}

	if ("png" %in% device) {
		grDevices::png(file.path(output_dir, "water_level.png"))
		on.exit(grDevices::dev.off())
		graphics::plot.new()
		plot_level(x)
		msg = sprintf(
			"Water level view save in %s",
			file.path(output_dir, "water_level.png")
		)
		return(msg)
	}
}
