#' Plot elementary damaging function
#' 
#' @param x array giving damaging function
#' @param d_select selected values of durations where to make some focus, at
#' least one value
#' @param h_range, height range for making the graphs. Default to NULL, and so
#' analyzing dam
#' @param d_range, duration range for making the graphs. Default to NULL, and so
#' analyzing dam
#' @param name, name of the elementary component processed. Default to NULL, no
#' title
#' @param output, path for saving the result. Default to NULL, nothing is saved,
#' graph is displayed
#' @param option, option on what is displayed ("all" or "total"). Default to
#' "all"
#' @param damaging_unit, unit of damaging
#' @param ... some extra parameters (unused)
#' @return NULL
#' 
#' @export

plot.elementary_damaging = function(
		x,
		d_select = getOption("floodam_building_d_select"),
		h_range = NULL,
		d_range = NULL,
		name = NULL,
		output = NULL,
		option = "all",
		damaging_unit = "euro",
		...
	)
{
	dam = x
	if (is.list(dam)) {
		if (is.null(name)) {name = names(dam[1])}
		dam = dam[[1]]
	}

	if (!option %in% c("all", "total")) option = "all"
	graphics::close.screen(all = TRUE)

	z = apply(dam, 1:2, sum, na.rm = TRUE)
	z[apply(dam, 1:2, function(y){all(is.na(y))})] = NA

	if (option == "total") {
		if (!is.null(output)) grDevices::pdf(file.path(output, sprintf("%s-total.pdf", name)), width = 15 , height = 10)
		plot_image_damaging (z, h_range, d_range)
		if (!is.null(output)) grDevices::dev.off()  
		return(invisible(NULL))
	}

	# General case (option = "all")
 	if (!is.null(output)) grDevices::pdf(file.path(output, sprintf("%s.pdf", name)), width = 15 , height = 10)
	
	# Prepare 4 different screens for general layout
	mat = rbind(c(0, 1, .9, 1), c(0, 1, .5, .95), c(0, 1, .07, .5), c(0, 1, 0, .1))
	graphics::split.screen(mat)
	  
	graphics::screen(1)
	if (!is.null(name)) {
		graphics::title(gsub("[_]", " ", name), cex.main = 1.5)
	}

	# Deal with general graphic, prepare 3 subscreens in screen 2
	mat = rbind(c(0, .45, 0, 1), c(0.45, .5, 0, 1), c(.5, 1, 0, 1))
	graphics::split.screen(mat, screen = 2)

	# Plot the total damage as an image
	graphics::screen(5)
	plot_image_damaging (z, h_range, d_range)
	
	# Plot the total damage, one line by duration
	graphics::screen(7)
	plot_line_damaging (z, d_select, damaging_unit)
	graphics::close.screen(5:7)
	
	# Plot damage by actions, one plot by duration
	graphics::split.screen(c(1, length(d_select)), screen = 3)
	n_action = dim(dam)[3]
	col = grDevices::rainbow(n_action)
	for (i in seq(length(d_select))) {
		graphics::screen(i + 4)
		plot(NA,
			xlim = range(as.integer(rownames(dam))), 
			ylim = range(apply(dam, 1:2, sum, na.rm = TRUE), 0), 
			xlab = "h (cm)", ylab = "damage (euro)",
			main = sprintf("Duration: %s hours", d_select[i])
		)
		
		for (a in seq(n_action)) {
			graphics::lines(as.integer(rownames(dam)), dam[ , as.character(d_select[i]), a], type = "l", col = col[a])
		}
	}
	
	# Plot legend of damage by actions, one legend for all
	graphics::screen(4)
	graphics::legend("center", legend = dimnames(dam)[[3]], lwd = 1, col = col, horiz = TRUE)
	
	graphics::close.screen(all = TRUE)
	if (!is.null(output)) grDevices::dev.off()
	
	invisible(NULL)
}
