#' plot a front perspective of a building
#' 
#' @param data_table, list of needed tables.
#' @param model_path, list of needed paths.
#' @param device, character, define way of plotting. Three options: "display", "pdf" or "png". Default to "display".
#' @param legend_table, data.frame, default to floodam.building::LEGEND
#' @param finition, character, define type of finishing for plotting. Two options: "check" or "presentation". Default to "check".
#' @param selection, character, selection of building to be plotted
#' @return A message that says what occurred
#' 
#' @keywords internal

plot_building_front_03 = function(
		data_table,
		model_path = NULL,
		device = "display",
		legend_table = floodam.building::LEGEND,
		finition = "check",
		selection = NULL
	)
{
	
	plot_vertical_surface = function(x_origin, x_final, H_abs, height, ...) {
		coordonate = list(
			x = c(x_origin, x_final, x_final, x_origin),
			y = H_abs + c(0, 0, height, height))
		graphics::polygon(coordonate, ...)
	}

	front_by_opening = function(opening, wall, col) {
		# Definition of points
		point = center = c(opening[["x"]], opening[["y"]])
		wall_a = c(wall[["x"]], wall[["y"]])
		wall_b = c(wall[["xx"]], wall[["yy"]])

		#	Projection of center on wall
		norm = (wall_b - wall_a) / sqrt(sum((wall_b - wall_a)^2))
		scalar = sum((center - wall_a) * norm)
		center = wall_a + sum((center - wall_a) * norm) * norm
		
		# Calculation of distance from origin of wall
		x_origin = sqrt((wall[["x"]] - center[1]) ^ 2 + (wall[["y"]] - center[2]) ^ 2) - opening[["width"]] / 200
		x_final = sqrt((wall[["x"]] - center[1]) ^ 2 + (wall[["y"]] - center[2]) ^ 2) + opening[["width"]] / 200

		plot_vertical_surface(wall[["x_origin"]] + x_origin, wall[["x_origin"]] + x_final, opening[["H_abs"]], opening[["height"]], col = col)
	}

	front_by_wall = function(wall, opening, coating) {

		if (!is.null(coating) && nrow(coating) > 0) {
			col = legend_table[coating[["coating_elementary_component"]], "col"]
			mapply(plot_vertical_surface, H_abs = coating[["H_abs"]], height = coating[["height"]], col = col, 
				MoreArgs = list(x_origin = wall[["x_origin"]], x_final = wall[["x_final"]], border = NA))
		}
		
		if (!is.null(opening) && nrow(opening) > 0) {
			if (any(is.na(opening[c("x","y")]))) opening[c("x","y")] = create_opening_coordinates(wall[c("x", "xx", "y", "yy")], nrow(opening))
			mapply(front_by_opening, split(opening, seq(nrow(opening))), col = "white", MoreArgs = list(wall = wall))
		}

		plot_vertical_surface(wall[["x_origin"]], wall[["x_final"]], wall[["H_abs"]], wall[["height"]])
		graphics::mtext(wall[["wall"]], side = 1, line = 0, outer = FALSE, at = mean(c(wall[["x_origin"]], wall[["x_final"]])))
	}

	front_by_room = function(wall, opening, coating, limit, i = NULL) {
		if (!is.null(i)) graphics::screen(i)
		room_name = if (finition == "presentation") unique(as.character(wall[["room_name"]])) else unique(as.character(wall[["room"]]))
		
		op = graphics::par(mar = c(2, 4, 1, 1) +.1)
		plot(x = NULL, y = NULL, xlab = "width", ylab = "height", xlim = c(limit[["x_min"]], limit[["x_max"]]), 
			ylim = c(limit[["y_min"]], limit[["y_max"]]), main = room_name)

		wall = cbind(wall, xx = c(wall[["x"]][-1], wall[["x"]][1]), yy = c(wall[["y"]][-1], wall[["y"]][1]))		
		wall[["x_origin"]] = c(0, cumsum(wall[["width"]])[-length(wall[["width"]])])
		wall[["x_final"]] = cumsum(wall[["width"]])

		level = levels(droplevels(wall[["wall"]]))
		wall = split_special(wall, "wall", level)
		opening = split_special(opening, "wall", level)
		coating = split_special(coating, "wall", level)
		mapply(front_by_wall, wall, opening, coating)
		graphics::par(op)
	}

	front_by_storey = function(storey, wall, opening, coating, output = NULL) {
		# Exclude external
		if (any(grep("external", wall[["room"]]))) wall = wall[-grep("external", wall[["room"]]), ]
		if (any(grep("external", opening[["room"]]))) opening = opening[-grep("external", opening[["room"]]), ]
		if (any(grep("external", coating[["room"]]))) coating = coating[-grep("external", coating[["room"]]), ]
		
		storey_name = if (finition == "presentation") as.character(storey[["name"]]) else as.character(storey[["storey"]])

		limit = list(
			x_min = 0,
			x_max = max(stats::aggregate(wall[["width"]], wall["room"], sum)[["x"]]),
			y_min = storey[["H_abs"]],
			y_max = storey[["height"]] + storey[["H_abs"]])

		level = levels(droplevels(wall[["room"]]))
		n = seq(length(level)) 
		room = split(n, (n - 1) %/% 8) 

		# Prepare different screens for general layout
		mat = as.matrix(data.frame(
			left = 0,
			right = 1,
			bottom = c(seq(.9, .1, length.out = 8 + 1), 0),
			up = c(1, seq(.9, .1, length.out = 8 + 1))
		))

		wall = split_special(wall, "room", level)
		opening = split_special(opening, "room", level)
		coating = split_special(coating, "room", level)
		
		graphics::close.screen(all.screens = TRUE)
		graphics::split.screen(mat)		
		mapply(front_by_room, 
			wall[room[[1]]], opening[room[[1]]], coating[room[[1]]], 
			seq(length(room[[1]])) + 1, MoreArgs = list(limit = limit))
		graphics::screen(1)
		graphics::title(main = storey_name)
		
		for (r in room[-1]) {
			graphics::close.screen(all.screens = TRUE)
			graphics::plot.new()
			graphics::split.screen(mat)					
			mapply(front_by_room, wall[r], opening[r], coating[r], seq(length(r)) + 1, MoreArgs = list(limit = limit))
			graphics::screen(1)
			graphics::title(main = storey_name)
		}
	}

	front_by_building = function(building, storey, wall, opening, coating) {
		building_name = if (finition == "presentation") as.character(building[["name"]]) else as.character(building[["building"]])

		level = levels(droplevels(wall[["storey"]]))
		storey = split_special(storey, "storey", level)
		wall = split_special(wall, "storey", level)
		opening = split_special(opening, "storey", level)
		coating = split_special(coating, "storey", level)

		if ("display" %in% device) {
			default_par = graphics::par(no.readonly = TRUE)
			mapply(front_by_storey, storey, wall, opening, coating, MoreArgs = list(output = NULL))
			graphics::par(default_par)
		}

		if ("pdf" %in% device) {
			file_name = file.path(model_path[["model_output_building_plan"]], sprintf("plan_front_%s.pdf", building[["building"]]))
			grDevices::pdf(file_name, 
				width = 15,
				height = 20)
			mapply(front_by_storey, storey, wall, opening, coating, MoreArgs = list(output = NULL))
			grDevices::dev.off()    
			return(sprintf("Front view of building plan save in %s", file_name))
		}
	}

	level = levels(data_table[["wall"]][["building"]])
 	building = split_special(data_table[["building"]], "building", level)
	storey = split_special(data_table[["storey"]], "building", level)
	wall = split_special(data_table[["wall"]], "building", level)
	opening = split_special(data_table[["opening"]], "building", level)
	coating = split_special(data_table[["coating"]], "building", level)

	if (is.null(selection)) selection = seq_along(building)
 	if (all(device == "display")) selection = selection[1]
 	mapply(front_by_building, building[selection], storey[selection], wall[selection], opening[selection], coating[selection])
	
}

plot_building_front_02 = function(
		data_table,
		model_path = NULL,
		device = "display",
		legend_table = floodam.building::LEGEND,
		finition = "check"
){
	
	plot_vertical_surface = function(x_origin, x_final, H_abs, height, ...) {
		coordonate = list(
			x = c(x_origin, x_final, x_final, x_origin),
			y = H_abs + c(0, 0, height, height))
		graphics::polygon(coordonate, ...)
	}

	front_by_opening = function(opening, wall, col) {
		# Definition of points
		point = center = c(opening[["x"]], opening[["y"]])
		wall_a = c(wall[["x"]], wall[["y"]])
		wall_b = c(wall[["xx"]], wall[["yy"]])

		#	Projection of center on wall
		norm = (wall_b - wall_a) / sqrt(sum((wall_b - wall_a)^2))
		scalar = sum((center - wall_a) * norm)
		center = wall_a + sum((center - wall_a) * norm) * norm
		
		# Calculation of distance from origin of wall
		x_origin = sqrt((wall[["x"]] - center[1]) ^ 2 + (wall[["y"]] - center[2]) ^ 2) - opening[["width"]] / 200
		x_final = sqrt((wall[["x"]] - center[1]) ^ 2 + (wall[["y"]] - center[2]) ^ 2) + opening[["width"]] / 200

		plot_vertical_surface(wall[["x_origin"]] + x_origin, wall[["x_origin"]] + x_final, opening[["H_abs"]], opening[["height"]], col = col)
	}

	front_by_wall = function(wall, opening, coating) {

		if (!is.null(coating) && nrow(coating) > 0) {
			col = legend_table[coating[["coating_elementary_component"]], "col"]
			mapply(plot_vertical_surface, H_abs = coating[["H_abs"]], height = coating[["height"]], col = col, 
				MoreArgs = list(x_origin = wall[["x_origin"]], x_final = wall[["x_final"]], border = NA))
		}
		
		if (!is.null(opening) && nrow(opening) > 0) {
			if (any(is.na(opening[c("x","y")]))) opening[c("x","y")] = create_opening_coordinates(wall[c("x", "xx", "y", "yy")], nrow(opening))
			mapply(front_by_opening, split(opening, seq(nrow(opening))), col = "white", MoreArgs = list(wall = wall))
		}

		plot_vertical_surface(wall[["x_origin"]], wall[["x_final"]], wall[["H_abs"]], wall[["height"]])
		graphics::mtext(wall[["wall"]], side = 1, line = 0, outer = FALSE, at = mean(c(wall[["x_origin"]], wall[["x_final"]])))
	}

	front_by_room = function(wall, opening, coating, limit, i = NULL) {
		if (!is.null(i)) graphics::screen(i)
		room_name = if (finition == "presentation") unique(as.character(wall[["room_name"]])) else unique(as.character(wall[["room"]]))
		
		op = graphics::par(mar = c(2, 4, 1, 1) +.1)
		plot(x = NULL, y = NULL, xlab = "width", ylab = "height", xlim = c(limit[["x_min"]], limit[["x_max"]]), 
			ylim = c(limit[["y_min"]], limit[["y_max"]]), main = room_name)

		wall = cbind(wall, xx = c(wall[["x"]][-1], wall[["x"]][1]), yy = c(wall[["y"]][-1], wall[["y"]][1]))		
		wall[["x_origin"]] = c(0, cumsum(wall[["width"]])[-length(wall[["width"]])])
		wall[["x_final"]] = cumsum(wall[["width"]])

		level = levels(droplevels(wall[["wall"]]))
		wall = split_special(wall, "wall", level)
		opening = split_special(opening, "wall", level)
		coating = split_special(coating, "wall", level)
		mapply(front_by_wall, wall, opening, coating)
		graphics::par(op)
	}

	front_by_storey = function(storey, wall, opening, coating, output = NULL) {
		# Exclude external
		if (any(grep("external", wall[["room"]]))) wall = wall[-grep("external", wall[["room"]]), ]
		if (any(grep("external", opening[["room"]]))) opening = opening[-grep("external", opening[["room"]]), ]
		if (any(grep("external", coating[["room"]]))) coating = coating[-grep("external", coating[["room"]]), ]
		
		storey_name = if (finition == "presentation") as.character(storey[["name"]]) else as.character(storey[["storey"]])

		limit = list(
			x_min = 0,
			x_max = max(stats::aggregate(wall[["width"]], wall["room"], sum)[["x"]]),
			y_min = storey[["H_abs"]],
			y_max = storey[["height"]] + storey[["H_abs"]])

		level = levels(droplevels(wall[["room"]]))
		n = seq(length(level)) 
		room = split(n, (n - 1) %/% 8) 

		# Prepare different screens for general layout
		mat = as.matrix(data.frame(
			left = 0,
			right = 1,
			bottom = c(seq(.9, .1, length.out = 8 + 1), 0),
			up = c(1, seq(.9, .1, length.out = 8 + 1))
		))

		wall = split_special(wall, "room", level)
		opening = split_special(opening, "room", level)
		coating = split_special(coating, "room", level)
		
		graphics::close.screen(all.screens = TRUE)
		graphics::split.screen(mat)		
		mapply(front_by_room, 
			wall[room[[1]]], opening[room[[1]]], coating[room[[1]]], 
			seq(length(room[[1]])) + 1, MoreArgs = list(limit = limit))
		graphics::screen(1)
		graphics::title(main = storey_name)
		
		for (r in room[-1]) {
			graphics::close.screen(all.screens = TRUE)
			graphics::plot.new()
			graphics::split.screen(mat)					
			mapply(front_by_room, wall[r], opening[r], coating[r], seq(length(r)) + 1, MoreArgs = list(limit = limit))
			graphics::screen(1)
			graphics::title(main = storey_name)
		}
	}

	# Preparing
	graphics::close.screen(all.screens = TRUE)

	storey = split_special(data_table[["storey"]], "storey", levels(data_table[["wall"]][["storey"]]))
	wall = split_special(data_table[["wall"]], "storey", levels(data_table[["wall"]][["storey"]]))
	opening = split_special(data_table[["opening"]], "storey", levels(data_table[["wall"]][["storey"]]))
	coating = split_special(data_table[["coating"]], "storey", levels(data_table[["wall"]][["storey"]]))

	if ("display" %in% device) {
		default_par = graphics::par(no.readonly = TRUE)
		mapply(front_by_storey, storey, wall, opening, coating, MoreArgs = list(output = NULL))
		graphics::par(default_par)
	}

	if ("pdf" %in% device) {
	grDevices::pdf(file.path(model_path[["model_output_building_plan"]], "building_front_plan.pdf"), 
			width = 15,
			height = 20)
		mapply(front_by_storey, storey, wall, opening, coating, MoreArgs = list(output = NULL))
	grDevices::dev.off()    
	return(sprintf("Front view of building plan save in %s", file.path(model_path[["model_output_building_plan"]], "building_front_plan.pdf")))
	}
}
