#' Identify rooms to merge when there's invisible walls
#'
#' @param model An object of class model
#' @param verbose boolean, will floodam tells what it is doing, default to
#' 		getOption("floodam_building_verbose")
#'
#' @description
#' Groups rooms that should be merged if they are separated by an invisible wall.
#' 
#' @return List of vectors of rooms to be merged

find_room_to_merge = function(
  model,
  verbose = getOption("floodam_building_verbose")
) {

  room = model[["data_table"]][["wall"]][c("name", "room", "room_name")]
  room[["room"]] = as.character(room[["room"]])
  room[["name"]] = as.character(room[["name"]])

  # Remove missing values
  room = room[stats::complete.cases(room), ]

  # Group room names by wall name
  match = split(room[["room"]], room[["name"]])

  # Merge connected room groups
  groups = list()
  for (connex in match) {
    found = FALSE
    for(i in seq(groups)) {
      if(any(connex %in% groups[[i]])) {
        groups[[i]] = unique(c(groups[[i]], connex))
        found = TRUE
      }
    }
    if(!found) {
      groups = append(groups, list(connex))
    }
  }

  if (verbose) {
    message(
      sprintf(
        "\t... found invisible walls between rooms for '%s'",
        model[["name"]]
      )
    )
  }

  return(groups)
}

#' merge rooms
#'
#' @param opening dataframe containing opening information
#' @param room dataframe containing room information
#' @param groups List of room vectors to be merged
#'
#' @description
#' - Merges rooms separated by invisible walls.
#' - Creates new room names and sums their areas.
#' - Updates openings to match the new merged room names.
#' 
#' @return List with updated `opening` and `room` dataframes

merge_room = function(opening, room, groups) {

  summed_room = list()

  for (g in groups) {
    new_name = paste(g, collapse = "-")

    # Update openings to reflect merged rooms
    opening[["down"]][opening[["down"]] %in% g] = new_name
    opening[["up"]][opening[["up"]] %in% g] = new_name

    room[["H_ceiling"]] = room[["H_abs"]] + room[["ceiling_H"]] / 100

    sel = room[room[["room"]] %in% g, ]

    # sum surfaces and volumes if rooms have same H_abs and ceiling_H
    agg = stats::aggregate(
      cbind(surface, v) ~ H_abs + ceiling_H + initial_depth + H_ceiling,
      data = sel,
      sum
    )

    # re order colnames
    agg = agg[c("surface", "H_abs", "ceiling_H", "initial_depth", "v",
      "H_ceiling")]

    # add room name
    agg = data.frame(room = new_name, agg, stringsAsFactors = FALSE)
    agg = agg[order(agg[["H_abs"]]), ]

    z = c(H_abs = unique(agg[["H_abs"]]), H_ceiling = unique(agg[["H_abs"]] +
      agg[["ceiling_H"]] / 100))

    # order this list
    z = sort(z)

    # only adding the lowest level
    mat = matrix(
      c(0, 0, min(z)),
      nrow = 1,
      dimnames = list(new_name, c("surface", "volume", "z"))
    )
    vol = 0.0

    # loop on all levels
    for(i in names(z)) {
      # height diff between two levels
      z_diff = z[[i]] - mat[nrow(mat), "z"]
      if (grepl("H_abs", i)) {
        # add surfaces and volumes when reaching new room
        pos = nrow(mat)
        sel = agg[agg[["H_abs"]] == z[[i]], ]
        vol = mat[nrow(mat), "volume"] + mat[nrow(mat), "surface"] * z_diff
        surf = mat[pos, "surface"] + sum(sel[["surface"]])
        mat = rbind(mat, c(surf, vol, z[[i]]))
      } else {
        # substract surfaces when reaching a ceiling and add volume
        pos = nrow(mat)
        sel = agg[agg[["H_ceiling"]] == z[[i]], ]
        vol = mat[nrow(mat), "volume"] + mat[nrow(mat), "surface"] * z_diff
        surf = mat[pos, "surface"] - sum(sel[["surface"]])
        # remove computing errors
        if (surf < 0.001) surf = 0
        mat = rbind(mat, c(surf, vol, z[[i]]))
      }
    }
    # clean matrix
    mat = mat[-1, , drop = FALSE]
    vmax = mat[nrow(mat), "volume"]
    mat = mat[-nrow(mat), , drop = FALSE]
    summed_room[[new_name]] = mat

    # remove old rooms
    room = room[!room[["room"]] %in% g, ]

    agg[1, "v"] = vmax
    # add new merged room
    room = rbind(room, agg[1, ])

  }

  return(list("opening" = opening, "room" = room, "summed_room" = summed_room))
}

#' split output
#'
#' @param out_hyd data.frame of hydraulic values: water exchanges through
#'  openings and floodwater depth in rooms
#' @param groups List of vectors of rooms to be merged
#' @param base_room Matrix containing informations on the rooms
#'
#' @description
#' Based on lists of rooms with invisible walls:
#' Copy the data for merged rooms in new branches to match the model rooms.
#' Sets water depth to the ground floor of each room.
#' 
#' @return data.frame of hydraulic values: water exchanges through openings and
#' 	floodwater depth in rooms

split_output = function(out_hyd, groups, base_room) {

  base_H = stats::setNames(base_room[["H_abs"]], base_room[["room"]])

  for (g in groups) {
    new_name = paste(g, collapse = "-")

    merged_col = out_hyd[, new_name, drop = FALSE]

    new_cols = sapply(g, function(r) {
      col_r = pmax(merged_col, base_H[r])
      col_r
    }, simplify = "matrix")
    colnames(new_cols) = g
    
    # add to the output
    out_hyd = cbind(out_hyd, new_cols)

    # removing merged room
    out_hyd = out_hyd[, colnames(out_hyd) != new_name, drop = FALSE]
  }
  
  return(out_hyd)
}