#' Check if all elementary components in a model has damaging functions.
#' 
#' @param data_table an object of with following items organized as tables: room, wall, coating, opening, furniture
#' @param file_log character, path of the log file
#' @param damaging, a list of elementary damaging functions, default to names(floodam.building::DAM)
#' @return NULL if everything is OK, or a list of elementary components for which no elementary damaging functions were found
#' 
#' @export


check_damaging = function(
		data_table,
		file_log,
		damaging = names(floodam.building::DAM)
	)
{
	which_missing = function(x, names = damaging) {
		if (is.null(x)) return(character())
		x = x[!is.na(x)]
		as.character(unique(x[!(x %in% names)]))
	}
	
	missing_component = c(
        which_missing(data_table[["room"]][["floor_elementary_component"]]),
        which_missing(data_table[["room"]][["floor_insulating_elementary_component"]]),
        which_missing(data_table[["room"]][["floor_coating_elementary_component"]]),
        which_missing(data_table[["room"]][["ceiling_elementary_component"]]),
        which_missing(data_table[["room"]][["ceiling_insulating_elementary_component"]]),
        which_missing(data_table[["room"]][["ceiling_coating_elementary_component"]]),
        which_missing(data_table[["wall"]][["wall_elementary_component"]]),
        which_missing(data_table[["wall"]][["insulating_elementary_component"]]),
        which_missing(data_table[["coating"]][["coating_elementary_component"]]),
        which_missing(data_table[["opening"]][["opening_elementary_component"]]),
        which_missing(data_table[["opening"]][["shutter_elementary_component"]]),
        which_missing(data_table[["furniture"]][["component_elementary"]])
	)
	
    check = length(missing_component)
	if (check > 0) {
		report = c(
			update_report(
                missing_component,
                sprintf(
                    "%s elementary component(s) without damaging functions:",
                    check
                )
            ),
			sprintf(
                "\nDamage estimated without %s elementary component(s)",
                check
            )
		)
		if (missing(file_log)) {
			warning(report)
		} else {
			write(report, file = file_log, append = TRUE)
			warning(
				sprintf(
					"%s elementary component(s) do not have damaging functions! Look at %s for details. Damage estimated without them.",
                    check,
					file_log
				)
			)
		}
		return(missing_component)
	} else {
		return(NULL)
	}
}

