#' Apply vulnerability reduction measures on the model
#
#' @param model the model to which measures will be applied
#' @param strategy_name the list of measures to apply
#' @param selection the rooms where measures will be applied
#' @param hazard_range list, default to getOption("floodam_building_hazard_range")
#' @param graph boolean, default to getOption("floodam_building_graph")
#' @param save_pdf boolean, default to FALSE
#' @param save_values boolean, default to FALSE
#' @return a list with all elements related to the implementation of the measures (the model
#' without and with measures, the list of the measures, the costs and the efficiency)
#' 
#' @export

apply_measure = function(
    model,
    strategy_name,
    selection,
    hazard_range = getOption("floodam_building_hazard_range"),
    graph = getOption("floodam_building_graph"),
    save_pdf = FALSE,
    save_values = FALSE
  )
{
	#Function to calcultate the new damages with the theme resist
  sill = function(x, y) {
		x[seq(y), 1:5] = 0
		x
	}	
	#Function to calculate the new damages with the theme avoid
	translate = function(damaging, h = 0) {
	  result = damaging
	  selection = as.integer(row.names(damaging)) < h
	  result[as.integer(row.names(damaging)) < h, ] = 0
	  result[as.integer(row.names(damaging)) >= h, ] = damaging[seq(nrow(damaging) - sum(selection)), ]
	  result
	}
	
	#Create a model to which adaptation measures will be applied
	model_adapted = model
	
  #Load the list of measures in the chosen strategy
  measure_name = list.files(file.path(model[["path"]][["adaptation"]], strategy_name), pattern = ".csv$")
  measure = lapply(file.path(model[["path"]][["adaptation"]], strategy_name, measure_name), utils::read.csv2, stringsAsFactors = FALSE)
  names(measure) = gsub(".csv$", "", measure_name)
  	
  
  #Apply all the changes in the XML and furniture due to the measures
  if ("material" %in% names(measure)) {
    model_adapted = modify_material(model_adapted, measure, selection)
 }

    
  #Calculate new damages
  #Step 1: compute the new damages due to the changes in the model
  model_adapted[["damaging"]] = compute_damage(model_adapted, hazard_range = hazard_range, verbose = FALSE)
  #Step 2 : if the measures "avoid" or "resist" are set, apply the global changes on the damages
  #Verify if "resist" is present and apply a sill corresponding to the dimensionning height of the measure
  if ("resist" %in% names(measure)) {
    ind_h_dim = which((as.numeric(rownames(model_adapted[["damaging"]][["absolute"]][["building"]]))) <= as.numeric(measure[["resist"]][["h_dim"]]), arr.ind = TRUE)
    model_adapted[["damaging"]][["absolute"]] = lapply(model_adapted[["damaging"]][["absolute"]], sill, y = max(ind_h_dim))
  }
  #Verify if "avoid" is present and translate the damage corresponding to the dimensionning height of the measure
  if ("avoid" %in% names(measure)) {
    model_adapted[["damaging"]][["absolute"]] = lapply(model_adapted[["damaging"]][["absolute"]], translate, h = measure[["avoid"]][["h_dim"]])
  }
  
  
  #Calculate costs
  measure_costs = compute_adaptation_costs(model, model_adapted, measure)
  
  
  #Calculate efficiency
  measure_efficiency = list(dam_diff = model[["damaging"]][["absolute"]], prevention = model[["damaging"]][["absolute"]], repair = model[["damaging"]][["absolute"]])
  for (a in names(model[["damaging"]][["absolute"]])) {
    measure_efficiency[["dam_diff"]][[a]] = model[["damaging"]][["absolute"]][[a]] - model_adapted[["damaging"]][["absolute"]][[a]]
    measure_efficiency[["prevention"]][[a]] = model[["damaging"]][["absolute"]][[a]] - model_adapted[["damaging"]][["absolute"]][[a]] - measure_costs[["prevention"]]
    measure_efficiency[["repair"]][[a]] = model[["damaging"]][["absolute"]][[a]] - model_adapted[["damaging"]][["absolute"]][[a]] - measure_costs[["repair"]]
  }

  #Plot results (new damages and efficiency)
  if (graph == TRUE){
    plot_efficiency(measure_efficiency[["prevention"]])
    plot_efficiency(measure_efficiency[["repair"]])
    plot(model_adapted, view = "damaging")
  }
  
  #Save efficiency and damaging plots in pdf
  if (save_pdf == TRUE || save_values == TRUE){
    #Create a file with the name of the strategy
    model_adapted[["path"]][["adaptation_output"]] = file.path(model[["path"]][["model_output_damaging"]], sprintf("%s-%s", strategy_name, selection[1]))
    if (!file.exists(model_adapted[["path"]][["adaptation_output"]])) {
      dir.create(path = model_adapted[["path"]][["adaptation_output"]])
    }
  }
  
  if (save_pdf == TRUE) {
    #Damaging model and model_adapted
    plot.damaging(
      model[["damaging"]][["absolute"]],
      path = model[["path"]][["model_output_damaging"]],
      file_name = sprintf("%s-damaging-model.pdf", model[["name"]])
    )
    plot.damaging(
      model_adapted[["damaging"]][["absolute"]],
      path = model_adapted[["path"]][["adaptation_output"]],
      file_name = sprintf("%s-damaging-model_adapted.pdf", model[["name"]])
    )
    #Efficiency damaging difference, prevention and repair
    plot.damaging(
      measure_efficiency[["dam_diff"]],
      path = model_adapted[["path"]][["adaptation_output"]],
      file_name = sprintf("%s-damaging-difference.pdf", model[["name"]])
    )
    plot_efficiency(
      measure_efficiency[["prevention"]],
      path = model_adapted[["path"]][["adaptation_output"]],
      file_name = sprintf("%s-efficiency-prevention.pdf", model[["name"]])
    )
    plot_efficiency(
      measure_efficiency[["repair"]],
      path = model_adapted[["path"]][["adaptation_output"]],
      file_name = sprintf("%s-efficiency-repair.pdf", model[["name"]])
    )
  }
   
  if (save_values == TRUE) {
    #Save efficiency tables of absolute[["total"]] into csv
    utils::write.csv2(measure_efficiency[["dam_diff"]][["total"]], file.path(model_adapted[["path"]][["adaptation_output"]], sprintf("%s-damaging-difference.csv", model[["name"]])))
    utils::write.csv2(measure_efficiency[["prevention"]][["total"]], file.path(model_adapted[["path"]][["adaptation_output"]], sprintf("%s-efficiency_prevention.csv", model[["name"]])))
    utils::write.csv2(measure_efficiency[["repair"]][["total"]], file.path(model_adapted[["path"]][["adaptation_output"]], sprintf("%s-efficiency_repair.csv", model[["name"]])))
    
    utils::write.csv2(measure_costs[["detail"]], file.path(model_adapted[["path"]][["adaptation_output"]], sprintf("%s-cost_detail.csv", model[["name"]])), row.names = FALSE)
  }
  
  #Save and return results of the model
  adaptation = list(
    model_adapted = model_adapted, 
    measure = measure,
    costs = measure_costs, 
    efficiency = measure_efficiency
    )
  return(adaptation)
}
