#' @title Analyse a model
#' 
#' @description 
#' This function performs a comprehensive analysis of a building model, 
#' allowing users to selectively execute different stages of the analysis 
#' pipeline, including loading the model, extracting information, calculating 
#' damages, generating hydraulic inputs, plotting results, and saving the model.
#' 
#' @param model either a character, name given to the model or an object of 
#' 	class model
#' @param type character, type of model that is loaded, default to ""
#' @param stage character, stage of the analysis to execute, default to
#' 	nothing; admitted values: "load", "extract", "damaging", "hydraulic",
#'  "graph", "save", and "display".  If "all" is specified, all available stages
#'  will be  executed. See details
#' @param hazard_range if damaging are calculated, gives the range for hazard,
#' 	default to getOption("floodam_building_hazard_range")
#' @param path list of useful paths, default to 
#' 	getOption("floodam_building_path")
#' @param file_name, list of useful file names, default to 
#' 	getOption("floodam_building_file_name")
#' @param type_building character, type of building file (building or yaml)
#' @param version_building character, version of building model to be used
#' @param verbose boolean, will floodam tells what it is doing, default to 
#' 	getOption("floodam_building_verbose")
#' 
#' @return an object of class model. See details
#' 
#' @details 
#' The `stage` parameter is a crucial parameter for controlling exactly which
#' parts of the analysis pipeline are executed. It is a character vector that
#' specifies the sequence of analysis steps (or "stages") to run:
#'
#' - "load" loads input file and creates object of class model. This is the 
#' 	 first stage to run
#' - "extract" extracts information from the model and organizes it in tables.
#' 	 This is the second stage you need to run and requires the "load" stage
#' 	 to have been run first
#' - "damaging" computes the damage function of the model. Requires the 
#' 	 "extract" stage
#' - "hydraulic" extracts hydraulic input data from model and computes damage
#'	 functions by room and segment of external wall (input in function 
#'   `analyse_hydraulics()`). Requires the "extract" stage
#' - "graph" saves plots in a temporary directory. The number of plots created 
#' 	 depends on the stage: Top and side view of building modeled and, if 
#' 	 "damaging" stage has been called already, the damaging function. 
#' 	 Requires the "extract" stage
#' - "save" saves data in the path stored in the model object created by the
#'   function. Requires the "extract" stage
#' - "display" displays plot of top view of the building model. Requires the
#' 	 "extract" stage
#' - "all" executes stages "load", "extract", "damaging", "hydraulic", "graph",
#' 	 "save" and "display"
#'  
#' The function returns a model object, which is a list with the following 
#' elements (the presence of each element depends on the stages executed):
#' 
#' - "name": a character string representing the model name
#' - "category": a character string representing the damage category
#' - "path": a list containing file paths related to the model
#' - "file_name": a list containing the file names related to the model
#' - "data_table": a list of data.frames containing information about the 
#'   building components and their attributes. Created/updated by the "extract"
#'   stage
#' - "value": a list of data.frames with information about (i) the monetary
#'   value of the building and its contents and (ii) the internal and external 
#'   surfaces calculated with the input data. Created by the "extract" stage
#' - "damaging": a list containing damage calculations. Created by the
#'   "damaging" stage
#' - "hydraulic": a list containing input data for a hydraulic model. Created by
#' 	 the "hydraulic" stage
#' - "dam_room_wall": a list containing damage calculations by room and wall 
#' 	 segment. Created by the "hydraulic" stage.
#' 
#' @examples
#' 
#' # Example 1
#' # declaring input and output paths
#' model_path = list(
#' 	data = system.file("extdata", package = "floodam.building"),
#' 	output = tempdir()
#' )
#' 
#' # analyzing model 'adu_t' of type 'adu' using stages *load*, *extract* and 
#' # *damaging* 
#' analyse_model(
#' 	model = "adu_t", 
#' 	type = "adu", 
#' 	stage = c("load", "extract", "damaging"), 
#' 	path = model_path
#' )
#' 
#' # Example 2
#' # using preloaded model 'adu_t_basement' 
#' model = adu_t_basement
#' 
#' # extracting model data
#' model = analyse_model(model = model, stage = "extract")
#' 
#' # calculating damage function
#' model = analyse_model(model = model, stage = "damaging")
#' 
#' # calculating hydraulic inputs
#' model = analyse_model(model = model, stage = "hydraulic")
#' 
#' @export

analyse_model = function(
		model,
		type = "",
		stage = "", 
		hazard_range = getOption("floodam_building_hazard_range"), 
		path = getOption("floodam_building_path"),
		file_name = getOption("floodam_building_file_name"),
		type_building = getOption("floodam_building_type_building"),
		version_building = NULL,
		verbose = getOption("floodam_building_verbose")
	)
{		
	start = Sys.time()
	if (identical(stage, "all")) stage = getOption("floodam_building_stage")
	stage = intersect(as.character(unlist(stage)), getOption("floodam_building_stage"))

    if (length(stage) == 0) {
        if (verbose) message("No admissible stage asked. Nothing is done.\n")
        return(NULL)
    }

	# Load model
	if ("load" %in% stage) {
		model = load_model(
			model_name = model,
			type = type,
			path = path,
			file_name = file_name,
			type_building = type_building,
			version_building = version_building,
			verbose = verbose
		)
	}

    if (! methods::is(model, "model")) {
        if (verbose)
            message(
                "At this stage 'model' should be a model but is not.",
                "Should you add 'load' in stage? Nothing is done."
            )
       return(NULL)
    }

	model_name = model[["name"]]
	file_log = file.path(
        model[["path"]][["model_log"]],
        model[["file_name"]][["log"]]
    )

	# Extracting information from building and furniture
	if ("extract" %in% stage) {
		model[["data_table"]] = extract_building_information(
			model,
			verbose
		)
		model[["value"]] = extract_value(
			room = model[["data_table"]][["room"]],
			furniture = model[["data_table"]][["furniture"]],
			category = model[["category"]],
			model_name = model_name,
			verbose = verbose
		)
	}

	# Calculate damaging
	if ("damaging" %in% stage) {
		model[["damaging"]] = compute_damage(
			model,
			hazard_range = hazard_range,
			verbose = verbose
		)
	}

	# Generate input data tables for hydraulic model
	if ("hydraulic" %in% stage) {
		model[["hydraulic"]] = extract_hydraulic_input(
			model,
			verbose = verbose
		)
		model[["dam_room_wall"]] = compute_damage_by_room_wall(
			model,
			hazard_range = hazard_range,
			verbose = verbose
		)
	}

	# Plotting graph damaging
	if ("graph" %in% stage) {
		if (verbose) message(sprintf("Plotting graphs for '%s'...", model_name))
		selected_view = c("top", "front")
		if (!is.null(model[["damaging"]])) selected_view = c(selected_view, "damaging")
		msg_plot = plot(model, view = selected_view, device = "pdf")
		if (verbose) message(paste("\t-", msg_plot))
	}

	# Saving model
	if ("save" %in% stage) {
		save_model(model, verbose)
	}

	# Plotting something in display
	if ("display" %in% stage) {
		if (verbose) message(sprintf("Display top view of '%s'", model_name))
		plot(model, view = "top", device = "display", finition = "check")
	}

	duration = Sys.time() - start

	if (verbose) {
		message(
			sprintf(
				"End of analysis for '%s'. Total elapsed time %.2f %s\nMore information availabe at %s\n",
				model_name,
				duration,
				attr(duration, "units"),
				file_log
			)
		)
	}
		
	invisible(model)
}

